<?php

namespace Drupal\alt_text_lab\Plugin\QueueWorker;

use Drupal\alt_text_lab\Service\AltTextGenerator;
use Drupal\alt_text_lab\Service\AltTextLabLogger;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\file\Entity\File;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes queued bulk alt text generation items.
 *
 * @QueueWorker(
 *    id = "alt_text_lab_bulk_queue",
 *    title = @Translation("Alt Text Lab bulk generation"),
 *    cron = {"time" = 60}
 *  )
 */
class AltTextLabBulkQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  protected AltTextGenerator $generator;
  protected AltTextLabLogger $logger;
  protected EntityTypeManagerInterface $entityTypeManager;
  protected Connection $database;
  protected TimeInterface $time;

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    AltTextGenerator $generator,
    AltTextLabLogger $logger,
    EntityTypeManagerInterface $entity_type_manager,
    Connection $database,
    TimeInterface $time,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->generator = $generator;
    $this->logger = $logger;
    $this->entityTypeManager = $entity_type_manager;
    $this->database = $database;
    $this->time = $time;
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('alt_text_lab.generator'),
      $container->get('alt_text_lab.logger'),
      $container->get('entity_type.manager'),
      $container->get('database'),
      $container->get('datetime.time'),
    );
  }

  public function processItem($data) {
    $fid = (int) ($data['fid'] ?? 0);
    $media_id = (int) ($data['media_id'] ?? 0);
    $field_name = (string) ($data['field_name'] ?? '');
    $delta = (int) ($data['delta'] ?? 0);
    $bulk_id = (int) ($data['bulk_id'] ?? 0);

    if (!$fid || !$media_id || $field_name === '') {
      return;
    }

    $file = File::load($fid);
    if (!$file) {
      $this->logger->logFailure($fid, 'File not found.', $bulk_id, $media_id, $field_name, $delta);
      return;
    }

    $result = $this->generator->generate($file);

    if ($result['error']) {
      $this->logger->logFailure($fid, $result['error'], $bulk_id, $media_id, $field_name, $delta);
      return;
    }

    $alt_text = $result['alt_text'];

    $media = $this->entityTypeManager->getStorage('media')->load($media_id);
    if ($media && $media->hasField($field_name) && !$media->get($field_name)->isEmpty()) {
      $field = $media->get($field_name);
      if (isset($field[$delta])) {
        $current_value = $field[$delta]->getValue();
        $current_value['alt'] = $alt_text;
        $field[$delta]->setValue($current_value);
      }

      try {
        $media->save();
      }
      catch (\Exception $e) {
        $this->logger->logFailure($fid, 'Failed to save media entity: ' . $e->getMessage(), $bulk_id, $media_id, $field_name, $delta);
        return;
      }
    }

    try {
      $this->database->insert('alt_text_lab_generation_history')
        ->fields([
          'fid' => $fid,
          'media_id' => $media_id,
          'alt_text' => $alt_text,
          'created' => $this->time->getRequestTime(),
          'bulk_id' => $bulk_id,
          'field_name' => $field_name,
          'delta' => $delta,
        ])
        ->execute();
    }
    catch (\Exception $e) {
      \Drupal::logger('alt_text_lab')->error('Failed to log generation history: @message', [
        '@message' => $e->getMessage(),
        'fid' => $fid,
        'media_id' => $media_id,
        'bulk_id' => $bulk_id,
      ]);
    }
  }

}






