<?php

namespace Drupal\alt_text_lab\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\file\FileInterface;

/**
 * Service for generating alt text for image files.
 *
 */
class AltTextGenerator {

  protected AltTextLabApi $api;
  protected AltTextLabImageValidator $imageValidator;

  protected ConfigFactoryInterface $configFactory;

  protected FileUrlGeneratorInterface $fileUrlGenerator;

  protected FileSystemInterface $fileSystem;

  public function __construct(
    AltTextLabApi $api,
    AltTextLabImageValidator $image_validator,
    ConfigFactoryInterface $config_factory,
    FileUrlGeneratorInterface $file_url_generator,
    FileSystemInterface $file_system,
  ) {
    $this->api = $api;
    $this->imageValidator = $image_validator;
    $this->configFactory = $config_factory;
    $this->fileUrlGenerator = $file_url_generator;
    $this->fileSystem = $file_system;
  }

  /**
   * Generate alt text for a file.
   */
  public function generate(FileInterface $file): array {
    // Check if it's an image.
    $mime_type = $file->getMimeType();
    if (!$mime_type || strpos($mime_type, 'image/') !== 0) {
      return [
        'alt_text' => NULL,
        'error' => 'File is not an image.',
        'error_code' => NULL,
      ];
    }

    // Additional validation.
    $validation = $this->imageValidator->validate($file);
    if (!$validation['valid']) {
      return [
        'alt_text' => NULL,
        'error' => (string) $validation['message'],
        'error_code' => NULL,
      ];
    }

    $config = $this->configFactory->get('alt_text_lab.settings');

    // Check API key.
    $api_key = (string) $config->get('api_key');
    if ($api_key === '') {
      return [
        'alt_text' => NULL,
        'error' => 'No API key configured.',
        'error_code' => NULL,
      ];
    }

    $file_uri = $file->getFileUri();

    // Build request payload.
    $payload = [
      'source' => 'drupal',
      'style' => $config->get('model_name') ?: NULL,
      'lang' => $config->get('lang') ?: 'en',
    ];

    $is_public = (bool) $config->get('is_public');

    if ($is_public) {
      $absolute_url = $this->fileUrlGenerator->generateAbsoluteString($file_uri);
      $payload['imageUrl'] = $absolute_url;
    }
    else {
      $realpath = $this->fileSystem->realpath($file_uri);
      $contents = $realpath ? @file_get_contents($realpath) : FALSE;
      if ($contents === FALSE) {
        return [
          'alt_text' => NULL,
          'error' => 'Failed to read file.',
          'error_code' => NULL,
        ];
      }
      $payload['imageRaw'] = base64_encode($contents);
    }

    [$data, $transport_error, $status] = $this->api->generateAltText($api_key, $payload);

    if ($transport_error) {
      return [
        'alt_text' => NULL,
        'error' => 'Request failed: ' . $transport_error,
        'error_code' => NULL,
      ];
    }

    // Check for specific API key errors.
    if ($data === 'API_KEY_IS_INVALID') {
      return [
        'alt_text' => NULL,
        'error' => 'API key is invalid!',
        'error_code' => 'API_KEY_IS_INVALID',
      ];
    }

    if ($data === 'NOT_ENOUGH_FUNDS') {
      return [
        'alt_text' => NULL,
        'error' => 'You don\'t have enough funds to generate alt text!',
        'error_code' => 'NOT_ENOUGH_FUNDS',
      ];
    }

    if (substr((string) $status, 0, 1) !== '2') {
      return [
        'alt_text' => NULL,
        'error' => 'Something went wrong.',
        'error_code' => NULL,
      ];
    }

    if (!is_array($data) || empty($data['result'])) {
      return [
        'alt_text' => NULL,
        'error' => 'No alt text in response.',
        'error_code' => NULL,
      ];
    }

    return [
      'alt_text' => (string) $data['result'],
      'error' => NULL,
      'error_code' => NULL,
    ];
  }

}




