<?php

namespace Drupal\alt_text_lab\Service;

use Drupal\alt_text_lab\Config\AltTextLabUrls;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Service wrapper around AltTextLab API.
 */
class AltTextLabApi {

  protected ClientInterface $httpClient;

  public function __construct(ClientInterface $http_client) {
    $this->httpClient = $http_client;
  }

  /**
   * Fetch account / subscription information.
   */
  public function getAccountInfo(string $api_key): array {

    try {
      $response = $this->httpClient->get(AltTextLabUrls::baseApiUrl() . '/v1/subscriptions/info', [
        'headers' => [
          'X-API-Key' => $api_key,
          'Content-Type' => 'application/json',
        ],
        'http_errors' => FALSE,
        'timeout' => 15,
      ]);
    } catch (GuzzleException $e) {
      return [NULL, $e->getMessage()];
    }

    $status = $response->getStatusCode();

    if (substr((string) $status, 0, 1) !== '2') {
      return [FALSE, NULL];
    }

    $body = (string) $response->getBody()->getContents();
    $data = json_decode($body, TRUE);

    if (!is_array($data)) {
      return [NULL, 'Unexpected API response format.'];
    }

    $account = [
      'plan' => $data['planName'] ?? NULL,
      'isActive' => (bool) ($data['isActive'] ?? FALSE),
      'credits' => (int) ($data['credits'] ?? 0),
      'nextReceiving' => $data['nextReceiving'] ?? NULL,
      'nextReceivingAt' => $data['nextReceivingAt'] ?? NULL,
    ];

    return [$account, NULL];
  }

  /**
   * Call alt text generation API.
   */
  public function generateAltText(string $api_key, array $payload): array {
    try {
      // Generation currently handled by a different host.
      $response = $this->httpClient->post(AltTextLabUrls::baseApiUrl() . '/v1/alt-text/generate', [
        'headers' => [
          'X-API-Key' => $api_key,
          'Content-Type' => 'application/json',
        ],
        'json' => $payload,
        'http_errors' => FALSE,
        'timeout' => 30,
      ]);
    }
    catch (GuzzleException $e) {
      return [NULL, $e->getMessage(), 0];
    }

    $status = $response->getStatusCode();
    $body = (string) $response->getBody()->getContents();
    $data = json_decode($body, TRUE);

    // Check for specific API errors.
    if ($status == 401) {
      return ['API_KEY_IS_INVALID', NULL, $status];
    }

    if ($status == 403 && isset($data['key']) && $data['key'] === 'NOT_ENOUGH_FUNDS') {
      return ['NOT_ENOUGH_FUNDS', NULL, $status];
    }

    return [$data, NULL, $status];
  }

}



