<?php

namespace Drupal\alt_text_lab\Service;

use Drupal\Core\File\FileSystemInterface;
use Drupal\file\FileInterface;

/**
 * Validates image files before sending them to Alt Text Lab API.
 */
class AltTextLabImageValidator {

  private const MAX_SIZE_BYTES = 16 * 1024 * 1024;

  private const MIN_WIDTH = 51;

  private const MIN_HEIGHT = 51;

  private array $validExtensions = ['jpg', 'jpeg', 'png', 'avif', 'webp', 'svg'];

  protected FileSystemInterface $fileSystem;

  public function __construct(FileSystemInterface $file_system) {
    $this->fileSystem = $file_system;
  }

  /**
   * Validate the given image file.
   */
  public function validate(FileInterface $file): array {
    // Extension check.
    $extension = strtolower(pathinfo($file->getFilename(), PATHINFO_EXTENSION));
    if (!in_array($extension, $this->validExtensions, TRUE)) {
      return [
        'valid' => FALSE,
        'message' => "Unsupported image type: {$extension}",
      ];
    }

    // File size check.
    $size = (int) $file->getSize();
    if ($size > self::MAX_SIZE_BYTES) {
      return [
        'valid' => FALSE,
        'message' => 'Image file size exceeds 16MB.',
      ];
    }

    // Dimensions check (for local files).
    $uri = $file->getFileUri();
    $realpath = $this->fileSystem->realpath($uri);

    if (!$realpath || !is_readable($realpath)) {
      return [
        'valid' => FALSE,
        'message' => 'File is not readable.',
      ];
    }

    $error_occurred = FALSE;
    $error_message = '';
    set_error_handler(function ($severity, $message, $file, $line) use (&$error_occurred, &$error_message) {
      $error_occurred = TRUE;
      $error_message = $message;
      return TRUE;
    });

    $info = getimagesize($realpath);

    restore_error_handler();

    if ($error_occurred || $info === FALSE || !isset($info[0], $info[1])) {
      return [
        'valid' => FALSE,
        'message' => $error_occurred ? 'Unable to read image dimensions: ' . $error_message : 'Unable to read image dimensions.',
      ];
    }

    $width = (int) $info[0];
    $height = (int) $info[1];

    if ($width < self::MIN_WIDTH || $height < self::MIN_HEIGHT) {
      return [
        'valid' => FALSE,
        'message' => "Image too small ({$width}x{$height}). Minimum size is 51x51.",
      ];
    }

    return [
      'valid' => TRUE,
      'message' => NULL,
    ];
  }

}














