(function (Drupal, drupalSettings) {
  function createEl(tag, className, text) {
    const el = document.createElement(tag);
    if (className) el.className = className;
    if (typeof text === 'string') el.textContent = text;
    return el;
  }

  function sanitize(str) {
    return (str || '').toString();
  }

  Drupal.behaviors.AiChatWidget = {
    attach(context) {
      const settings = drupalSettings.ai_chat;
      if (!settings || context !== document) return;

      if (document.getElementById('ai-chat-toggle')) {
        return; // Already added.
      }

      const assistantId = settings.assistant;
      const storageKey = `ai_chat_thread_${assistantId}`;
      let threadId = localStorage.getItem(storageKey) || '';

      // Toggle button
      const toggle = createEl('button', 'ai-chat-toggle');
      toggle.type = 'button';
      toggle.id = 'ai-chat-toggle';
      toggle.setAttribute('aria-expanded', 'false');
      toggle.setAttribute('aria-label', 'Open chat');
      toggle.title = 'Chat';
      toggle.textContent = '💬';

      // Chat panel
      const panel = createEl('div', 'ai-chat-panel');
      panel.id = 'ai-chat-panel';
      panel.setAttribute('hidden', 'hidden');
      toggle.setAttribute('aria-controls', panel.id);

      const header = createEl('div', 'ai-chat-header');
      header.appendChild(createEl('div', 'ai-chat-title', 'AI Assistant'));
      const closeBtn = createEl('button', 'ai-chat-close', '×');
      closeBtn.type = 'button';
      closeBtn.setAttribute('aria-label', 'Close');
      header.appendChild(closeBtn);

      const messages = createEl('div', 'ai-chat-messages');

      const form = createEl('form', 'ai-chat-form');
      const input = createEl('input', 'ai-chat-input');
      input.type = 'text';
      input.placeholder = 'Type your message...';
      input.required = true;
      const sendBtn = createEl('button', 'ai-chat-send');
      sendBtn.type = 'submit';
      sendBtn.textContent = 'Send';
      form.appendChild(input);
      form.appendChild(sendBtn);

      panel.appendChild(header);
      panel.appendChild(messages);
      panel.appendChild(form);

      document.body.appendChild(toggle);
      document.body.appendChild(panel);

      const openKey = `ai_chat_open_${assistantId}`;

      function setOpen(open, focusInput = false) {
        if (open === true) {
          panel.style.display = 'block';
          toggle.setAttribute('aria-expanded', 'true');
          toggle.setAttribute('aria-label', 'Close chat');
          localStorage.setItem(openKey, '1');
          if (focusInput) {
            input.focus();
          }
        } else {
          // Set display none on panel
          panel.style.display = 'none';
          toggle.setAttribute('aria-expanded', 'false');
          toggle.setAttribute('aria-label', 'Open chat');
          localStorage.setItem(openKey, '0');
        }
      }

      // Restore open state on load.
      (function initOpenState() {
        const saved = localStorage.getItem(openKey);
        const shouldOpen = saved === '1';
        setOpen(shouldOpen, false);
      })();

      toggle.addEventListener('click', function () {
        const isOpen = !panel.style.display || panel.style.display === 'block';
        setOpen(!isOpen, !isOpen);
      });
      closeBtn.addEventListener('click', function () {
        setOpen(false);
      });

      function appendMessage(role, text) {
        const wrap = createEl('div', `ai-chat-msg ai-chat-msg-${role}`);
        wrap.appendChild(createEl('div', 'ai-chat-bubble', sanitize(text)));
        messages.appendChild(wrap);
        messages.scrollTop = messages.scrollHeight;
      }

      async function sendMessage(text) {
        const endpoint = settings.endpoint;
        const payload = { message: text };
        if (threadId) payload.thread_id = threadId;
        appendMessage('user', text);

        // Loading placeholder
        const loading = createEl('div', 'ai-chat-msg ai-chat-msg-assistant');
        const bubble = createEl('div', 'ai-chat-bubble', '...');
        loading.appendChild(bubble);
        messages.appendChild(loading);
        messages.scrollTop = messages.scrollHeight;

        try {
          const resp = await fetch(endpoint, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            credentials: 'same-origin',
            body: JSON.stringify(payload),
          });
          const data = await resp.json();
          loading.remove();
          if (data && data.ok) {
            if (data.thread_id) {
              threadId = data.thread_id;
              localStorage.setItem(storageKey, threadId);
            }
            appendMessage('assistant', data.reply || '');
          } else {
            appendMessage(
              'assistant',
              data && data.error ? `Error: ${data.error}` : 'Error',
            );
          }
        } catch (e) {
          loading.remove();
          appendMessage('assistant', 'Network error');
        }
      }

      form.addEventListener('submit', function (e) {
        e.preventDefault();
        const text = input.value.trim();
        if (!text) return;
        input.value = '';
        sendMessage(text);
      });
    },
  };
})(Drupal, drupalSettings);
