<?php

namespace Drupal\ai_chat\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\ai_assistant_api\Data\UserMessage;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for AI chat interactions.
 */
class ChatController extends ControllerBase {

  /**
   * The AI Assistant runner service.
   *
   * @var mixed
   */
  protected $runner;

  /**
   * Creates an instance of the controller.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   *
   * @return static
   *   The constructed controller.
   */
  public static function create(ContainerInterface $container): static {
    $instance = new static();
    $instance->runner = $container->get('ai_assistant_api.runner');
    return $instance;
  }

  /**
   * AJAX endpoint to send a message to the configured assistant.
   */
  public function send(Request $request): JsonResponse {
    $config = $this->config('ai_chat.settings');
    $assistant_id = $config->get('assistant');
    $allowed_roles = array_filter($config->get('allowed_roles') ?: []);

    if (!$assistant_id || empty($allowed_roles)) {
      return new JsonResponse(['ok' => FALSE, 'error' => 'Not configured.'], 400);
    }

    $account = $this->currentUser();
    $has_access = FALSE;
    foreach ($allowed_roles as $rid) {
      if ($account->hasPermission('administer site configuration') || in_array($rid, $account->getRoles(), TRUE)) {
        $has_access = TRUE;
        break;
      }
    }
    if (!$has_access) {
      return new JsonResponse(['ok' => FALSE, 'error' => 'Access denied.'], 403);
    }

    $data = json_decode($request->getContent() ?: '[]', TRUE) ?: [];
    $message = trim((string) ($data['message'] ?? ''));
    $thread_id = isset($data['thread_id']) ? (string) $data['thread_id'] : '';

    if ($message === '') {
      return new JsonResponse(['ok' => FALSE, 'error' => 'Empty message.'], 400);
    }

    $assistant = $this->entityTypeManager()->getStorage('ai_assistant')->load($assistant_id);
    if (!$assistant) {
      return new JsonResponse(['ok' => FALSE, 'error' => 'Assistant not found.'], 400);
    }

    try {
      $this->runner->setAssistant($assistant);
      if (!empty($thread_id)) {
        $this->runner->setThreadsKey($thread_id);
      }
      $this->runner->setUserMessage(new UserMessage($message));
      $output = $this->runner->process();

      $normalized = $output->getNormalized();
      if (is_object($normalized) && method_exists($normalized, 'getText')) {
        $reply = (string) $normalized->getText();
      }
      else {
        $reply = (string) $normalized;
      }
      $current_thread = $this->runner->getThreadsKey();

      return new JsonResponse([
        'ok' => TRUE,
        'reply' => $reply,
        'thread_id' => $current_thread,
      ]);
    }
    catch (\Throwable $e) {
      return new JsonResponse([
        'ok' => FALSE,
        'error' => $e->getMessage(),
      ], 500);
    }
  }

}
