<?php

namespace Drupal\ai_chat\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Settings form for AI Chat.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * Entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   *
   * @return string[]
   *   The list of editable configuration names.
   */
  protected function getEditableConfigNames(): array {
    return ['ai_chat.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ai_chat_settings_form';
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array<string, mixed>
   *   The form array with added elements.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('ai_chat.settings');

    // Roles options.
    /** @var \Drupal\user\RoleInterface[] $roles */
    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    $role_options = [];
    foreach ($roles as $rid => $role) {
      $role_options[$rid] = $role->label();
    }

    // Assistants options.
    $assistants = $this->entityTypeManager->getStorage('ai_assistant')->loadMultiple();
    $assistant_options = [];
    foreach ($assistants as $id => $assistant) {
      $assistant_options[$id] = $assistant->label();
    }

    $form['allowed_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Roles allowed to access chat'),
      '#description' => $this->t('Users with any of the selected roles will see and can use the floating AI chat.'),
      '#options' => $role_options,
      '#default_value' => $config->get('allowed_roles') ?: [],
    ];

    $form['assistant'] = [
      '#type' => 'select',
      '#title' => $this->t('AI Assistant'),
      '#description' => $this->t('Select the AI Assistant (from ai_assistant_api) to use for this chat.'),
      '#options' => $assistant_options,
      '#empty_option' => $this->t('- Select -'),
      '#default_value' => $config->get('assistant') ?: NULL,
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);
    $allowed_roles = array_values(array_filter($form_state->getValue('allowed_roles') ?? []));
    $this->configFactory()->getEditable('ai_chat.settings')
      ->set('allowed_roles', $allowed_roles)
      ->set('assistant', $form_state->getValue('assistant'))
      ->save();
  }

}
