<?php

namespace Drupal\ai_chat\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Settings form for AI Chat.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * Entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   *
   * @return string[]
   *   The list of editable configuration names.
   */
  protected function getEditableConfigNames(): array {
    return ['ai_chat.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ai_chat_settings_form';
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array<string, mixed>
   *   The form array with added elements.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('ai_chat.settings');

    // Roles options.
    /** @var \Drupal\user\RoleInterface[] $roles */
    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    $role_options = [];
    foreach ($roles as $rid => $role) {
      $role_options[$rid] = $role->label();
    }

    // Assistants options.
    $assistants = $this->entityTypeManager->getStorage('ai_assistant')->loadMultiple();
    $assistant_options = [];
    foreach ($assistants as $id => $assistant) {
      $assistant_options[$id] = $assistant->label();
    }

    $form['description'] = [
      '#markup' => '<p>' . $this->t('Configure AI chat assistants for each role. Users will only see the chat if they have at least one role with an assigned assistant.') . '</p>',
    ];

    // Get current role assistants configuration.
    $role_assistants = $config->get('role_assistants') ?: [];

    $form['role_assistants'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Role-based Assistant Configuration'),
      '#tree' => TRUE,
    ];

    foreach ($roles as $rid => $role) {
      $form['role_assistants'][$rid] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Role: @role', ['@role' => $role->label()]),
        '#collapsible' => TRUE,
        '#collapsed' => empty($role_assistants[$rid]['assistants']),
      ];

      $form['role_assistants'][$rid]['enabled'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable AI chat for this role'),
        '#default_value' => !empty($role_assistants[$rid]['assistants']),
      ];

      $form['role_assistants'][$rid]['assistant'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('AI Assistants'),
        '#description' => $this->t('Select one or more AI Assistants to use for users with this role. Users will be able to choose between them.'),
        '#options' => $assistant_options,
        '#default_value' => $role_assistants[$rid]['assistants'] ?? [],
        '#states' => [
          'visible' => [
            ':input[name="role_assistants[' . $rid . '][enabled]"]' => ['checked' => TRUE],
          ],
          'required' => [
            ':input[name="role_assistants[' . $rid . '][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form['role_assistants'][$rid]['default_assistant'] = [
        '#type' => 'select',
        '#title' => $this->t('Default assistant'),
        '#description' => $this->t('Select the default assistant for this role. If not set, "Default" will be used.'),
        '#options' => ['' => $this->t('Default')] + $assistant_options,
        '#default_value' => $role_assistants[$rid]['default_assistant'] ?? '',
        '#required' => FALSE,
        '#states' => [
          'visible' => [
            ':input[name="role_assistants[' . $rid . '][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);
    $role_assistants = [];
    foreach ($form_state->getValue('role_assistants') as $rid => $assistant_config) {
      if (!empty($assistant_config['enabled']) && !empty($assistant_config['assistant'])) {
        $filtered_assistants = array_filter($assistant_config['assistant']);
        if (!empty($filtered_assistants)) {
          $role_assistants[$rid] = [
            'assistants' => array_values($filtered_assistants),
            'default_assistant' => $assistant_config['default_assistant'] ?? '',
          ];
        }
      }
    }
    $this->configFactory()->getEditable('ai_chat.settings')
      ->set('role_assistants', $role_assistants)
      ->save();
  }

}
