(function (Drupal, drupalSettings) {
  /**
   * Create a DOM element with optional class and text content.
   *
   * @param {string} tag - The HTML tag name.
   * @param {string|null} className - The CSS class name.
   * @param {string|null} text - The text content.
   * @returns {HTMLElement} The created element.
   */
  function createEl(tag, className, text) {
    const el = document.createElement(tag);
    if (className) el.className = className;
    if (typeof text === 'string') el.textContent = text;
    return el;
  }

  /**
   * Convert URLs in text to clickable links.
   *
   * @param {string} text - The input text.
   * @returns {string} The text with URLs converted to links.
   */
  function convertLinksToClickable(text) {
    const urlRegex = /(https?:\/\/[^\s<>"']+)/g;
    const div = document.createElement('div');
    div.textContent = text;
    let escapedText = div.innerHTML;
    escapedText = escapedText.replace(
      urlRegex,
      '<a href="$1" target="_blank" rel="noopener noreferrer">$1</a>',
    );

    return escapedText;
  }

  /**
   * AI Chat Widget class.
   */
  class AiChatWidget {
    /**
     * Constructor.
     *
     * @param {Object} assistants - Available assistants.
     * @param {string} currentAssistant - Current assistant ID.
     * @param {string} endpoint - API endpoint.
     */
    constructor(assistants, currentAssistant, endpoint) {
      this.assistants = assistants;
      this.currentAssistant = currentAssistant;
      this.endpoint = endpoint;
      this.threadId = '';
      this.onAssistantChange = null;

      // Storage keys
      this.openKey = `ai_chat_open_${this.currentAssistant}`;
      this.posKey = `ai_chat_pos_${this.currentAssistant}`;

      // DOM elements
      this.toggle = null;
      this.panel = null;
      this.messages = null;
      this.input = null;
      this.assistantSelect = null;
    }

    /**
     * Initialize the widget.
     */
    init() {
      this.createElements();
      this.attachEventListeners();
      this.restoreState();
    }

    /**
     * Create DOM elements.
     */
    createElements() {
      this.toggle = createEl('button', 'ai-chat-toggle');
      this.toggle.type = 'button';
      this.toggle.id = 'ai-chat-toggle';
      this.toggle.setAttribute('aria-expanded', 'false');
      this.toggle.setAttribute('aria-label', 'Open chat');
      this.toggle.title = 'Chat';
      this.toggle.textContent = '💬';
      this.panel = createEl('div', 'ai-chat-panel');
      this.panel.id = 'ai-chat-panel';
      this.panel.setAttribute('hidden', 'hidden');
      this.toggle.setAttribute('aria-controls', this.panel.id);
      const header = this.createHeader();
      this.messages = createEl('div', 'ai-chat-messages');
      const form = this.createForm();
      this.panel.appendChild(header);
      this.panel.appendChild(this.messages);
      this.panel.appendChild(form);
      document.body.appendChild(this.toggle);
      document.body.appendChild(this.panel);
    }

    /**
     * Create header with title and assistant selector.
     *
     * @returns {HTMLElement} The header element.
     */
    createHeader() {
      const header = createEl('div', 'ai-chat-header');
      const titleContainer = createEl('div', 'ai-chat-title-container');
      const title = createEl('div', 'ai-chat-title', 'AI Assistant');
      titleContainer.appendChild(title);

      // Create assistant selector if multiple assistants
      if (Object.keys(this.assistants).length > 1) {
        this.assistantSelect = createEl('select', 'ai-chat-assistant-select');
        this.assistantSelect.setAttribute('aria-label', 'Select AI Assistant');

        Object.entries(this.assistants).forEach(([id, label]) => {
          const option = createEl('option', '', label);
          option.value = id;
          if (id === this.currentAssistant) {
            option.selected = true;
          }
          this.assistantSelect.appendChild(option);
        });

        titleContainer.appendChild(this.assistantSelect);
      }

      header.appendChild(titleContainer);
      const closeBtn = createEl('button', 'ai-chat-close', '×');
      closeBtn.type = 'button';
      closeBtn.setAttribute('aria-label', 'Close');
      header.appendChild(closeBtn);

      return header;
    }

    /**
     * Create the form with input and send button.
     *
     * @returns {HTMLElement} The form element.
     */
    createForm() {
      const form = createEl('form', 'ai-chat-form');
      this.input = createEl('textarea', 'ai-chat-input');
      this.input.setAttribute('rows', '2');
      this.input.placeholder = 'Type your message...';
      this.input.required = true;

      const sendBtn = createEl('button', 'ai-chat-send');
      sendBtn.type = 'submit';
      sendBtn.textContent = 'Send';

      form.appendChild(this.input);
      form.appendChild(sendBtn);

      return form;
    }

    /**
     * Attach event listeners.
     */
    attachEventListeners() {
      this.toggle.addEventListener('click', () => {
        const isOpen =
          !this.panel.hasAttribute('hidden') &&
          this.panel.style.display === 'block';
        this.setOpen(!isOpen, !isOpen);
      });

      const closeBtn = this.panel.querySelector('.ai-chat-close');
      closeBtn.addEventListener('click', () => {
        this.setOpen(false);
      });

      if (this.assistantSelect) {
        this.assistantSelect.addEventListener('change', (event) => {
          const newAssistant = event.target.value;
          this.changeAssistant(newAssistant);
        });
      }

      const form = this.panel.querySelector('.ai-chat-form');
      form.addEventListener('submit', (event) => {
        event.preventDefault();
        const text = this.input.value.trim();
        if (text) {
          this.input.value = '';
          this.sendMessage(text);
        }
      });

      // Keyboard shortcuts
      this.input.addEventListener('keydown', (event) => {
        if (event.key === 'Enter' && !event.shiftKey) {
          event.preventDefault();
          if (this.input.value.trim()) {
            form.requestSubmit();
          }
        }
      });

      // Drag functionality
      this.enableDragToMove();

      // Window resize
      window.addEventListener('resize', () => {
        this.clampPosition();
      });
    }

    /**
     * Change the current assistant.
     *
     * @param {string} newAssistant - The new assistant ID.
     */
    changeAssistant(newAssistant) {
      if (newAssistant !== this.currentAssistant) {
        this.messages.innerHTML = '';
        this.currentAssistant = newAssistant;
        this.updateStorageKeys();
        this.threadId =
          localStorage.getItem(this.getStorageKey('thread')) || '';

        if (this.onAssistantChange) {
          this.onAssistantChange(newAssistant);
        }
      }
    }

    /**
     * Update storage keys for current assistant.
     */
    updateStorageKeys() {
      this.openKey = `ai_chat_open_${this.currentAssistant}`;
      this.posKey = `ai_chat_pos_${this.currentAssistant}`;
    }

    /**
     * Get storage key with suffix.
     *
     * @param {string} suffix - The suffix for the key.
     * @returns {string} The storage key.
     */
    getStorageKey(suffix) {
      return `ai_chat_${suffix}_${this.currentAssistant}`;
    }

    /**
     * Set the open state of the chat.
     *
     * @param {boolean} open - Whether to open the chat.
     * @param {boolean} focusInput - Whether to focus the input.
     */
    setOpen(open, focusInput = false) {
      if (open) {
        this.panel.style.display = 'block';
        this.panel.removeAttribute('hidden');
        this.toggle.setAttribute('aria-expanded', 'true');
        this.toggle.setAttribute('aria-label', 'Close chat');
        localStorage.setItem(this.openKey, '1');
        if (focusInput) {
          this.input.focus();
        }
      } else {
        this.panel.style.display = 'none';
        this.panel.setAttribute('hidden', 'hidden');
        this.toggle.setAttribute('aria-expanded', 'false');
        this.toggle.setAttribute('aria-label', 'Open chat');
        localStorage.setItem(this.openKey, '0');
      }
    }

    /**
     * Restore the widget state from localStorage.
     */
    restoreState() {
      this.restorePosition();

      const saved = localStorage.getItem(this.openKey);
      const shouldOpen = saved === '1';
      this.setOpen(shouldOpen, false);
    }

    /**
     * Restore position from localStorage.
     */
    restorePosition() {
      try {
        const raw = localStorage.getItem(this.posKey);
        if (!raw) return;

        const pos = JSON.parse(raw);
        if (typeof pos.left === 'number' && typeof pos.top === 'number') {
          const clamped = this.clampToViewport(pos.left, pos.top);
          this.panel.style.left = `${clamped.left}px`;
          this.panel.style.top = `${clamped.top}px`;
          this.panel.style.right = 'auto';
          this.panel.style.bottom = 'auto';
        }
      } catch (e) {
        // Ignore localStorage errors
      }
    }

    /**
     * Clamp position to viewport bounds.
     *
     * @param {number} left - Left position.
     * @param {number} top - Top position.
     * @returns {Object} Clamped position.
     */
    clampToViewport(left, top) {
      const margin = 8;
      const vpW = window.innerWidth;
      const vpH = window.innerHeight;
      const rect = this.panel.getBoundingClientRect();
      const width = rect.width;
      const height = rect.height;
      const clampedLeft = Math.max(
        margin,
        Math.min(left, vpW - width - margin),
      );
      const clampedTop = Math.max(margin, Math.min(top, vpH - height - margin));

      return { left: clampedLeft, top: clampedTop };
    }

    /**
     * Clamp current position and save it.
     */
    clampPosition() {
      const rect = this.panel.getBoundingClientRect();
      const clamped = this.clampToViewport(rect.left, rect.top);
      this.panel.style.left = `${clamped.left}px`;
      this.panel.style.top = `${clamped.top}px`;
      this.savePosition(clamped.left, clamped.top);
    }

    /**
     * Save position to localStorage.
     *
     * @param {number} left - Left position.
     * @param {number} top - Top position.
     */
    savePosition(left, top) {
      try {
        localStorage.setItem(
          this.posKey,
          JSON.stringify({
            left: Math.round(left),
            top: Math.round(top),
          }),
        );
      } catch (e) {
        // Ignore localStorage errors
      }
    }

    /**
     * Enable drag to move functionality.
     */
    enableDragToMove() {
      let dragging = false;
      let startX = 0;
      let startY = 0;
      let startLeft = 0;
      let startTop = 0;

      const onMouseMove = (e) => {
        if (!dragging) return;

        const dx = e.clientX - startX;
        const dy = e.clientY - startY;

        const newLeft = startLeft + dx;
        const newTop = startTop + dy;

        const clamped = this.clampToViewport(newLeft, newTop);
        this.panel.style.left = `${clamped.left}px`;
        this.panel.style.top = `${clamped.top}px`;
      };

      const endDrag = () => {
        if (!dragging) return;

        dragging = false;
        document.removeEventListener('mousemove', onMouseMove);
        document.removeEventListener('mouseup', endDrag);

        this.clampPosition();
      };

      const header = this.panel.querySelector('.ai-chat-header');
      const closeBtn = this.panel.querySelector('.ai-chat-close');

      header.addEventListener('mousedown', (e) => {
        // Ignore clicks on the close button
        if (e.target === closeBtn || closeBtn.contains(e.target)) {
          return;
        }

        dragging = true;
        const rect = this.panel.getBoundingClientRect();
        startX = e.clientX;
        startY = e.clientY;
        startLeft = rect.left;
        startTop = rect.top;

        // Switch to left/top positioning for free movement
        this.panel.style.left = `${startLeft}px`;
        this.panel.style.top = `${startTop}px`;
        this.panel.style.right = 'auto';
        this.panel.style.bottom = 'auto';

        document.addEventListener('mousemove', onMouseMove);
        document.addEventListener('mouseup', endDrag);
      });
    }

    /**
     * Append a message to the chat.
     *
     * @param {string} role - The message role ('user' or 'assistant').
     * @param {string} text - The message text.
     */
    appendMessage(role, text) {
      const wrap = createEl('div', `ai-chat-msg ai-chat-msg-${role}`);
      const bubble = createEl('div', 'ai-chat-bubble');
      bubble.innerHTML = convertLinksToClickable(text);
      wrap.appendChild(bubble);
      this.messages.appendChild(wrap);
      this.messages.scrollTop = this.messages.scrollHeight;
    }

    /**
     * Send a message to the AI assistant.
     *
     * @param {string} text - The message text.
     */
    async sendMessage(text) {
      const payload = {
        message: text,
        assistant_id: this.currentAssistant,
      };

      if (this.threadId) {
        payload.thread_id = this.threadId;
      }

      this.appendMessage('user', text);
      const loading = createEl('div', 'ai-chat-msg ai-chat-msg-assistant');
      const bubble = createEl('div', 'ai-chat-bubble', '...');
      loading.appendChild(bubble);
      this.messages.appendChild(loading);
      this.messages.scrollTop = this.messages.scrollHeight;

      try {
        const response = await fetch(this.endpoint, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          credentials: 'same-origin',
          body: JSON.stringify(payload),
        });

        const data = await response.json();
        loading.remove();

        if (data && data.ok) {
          if (data.thread_id) {
            this.threadId = data.thread_id;
            localStorage.setItem(this.getStorageKey('thread'), this.threadId);
          }
          this.appendMessage('assistant', data.reply || '');
        } else {
          const errorMsg =
            data && data.error ? `Error: ${data.error}` : 'Error';
          this.appendMessage('assistant', errorMsg);
        }
      } catch (error) {
        loading.remove();
        this.appendMessage('assistant', 'Network error');
        console.error('AI Chat error:', error);
      }
    }
  }

  /**
   * AI Chat Widget behavior.
   */
  Drupal.behaviors.AiChatWidget = {
    attach(context) {
      const settings = drupalSettings.ai_chat;
      if (!settings || context !== document) return;

      if (document.getElementById('ai-chat-toggle')) {
        return;
      }

      const assistants = settings.assistants || {};
      const defaultAssistant = settings.defaultAssistant;
      let currentAssistant = defaultAssistant;

      function getStorageKey(suffix) {
        return `ai_chat_${suffix}_${currentAssistant}`;
      }

      let threadId = localStorage.getItem(getStorageKey('thread')) || '';

      // Create and setup the chat widget
      const widget = new AiChatWidget(
        assistants,
        currentAssistant,
        settings.endpoint,
      );
      widget.init();

      // Handle assistant change
      widget.onAssistantChange = function (newAssistant) {
        if (newAssistant !== currentAssistant) {
          currentAssistant = newAssistant;
          threadId = localStorage.getItem(getStorageKey('thread')) || '';
        }
      };
    },
  };
})(Drupal, drupalSettings);
