<?php

namespace Drupal\ai_chat\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\ai_assistant_api\Data\UserMessage;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for AI chat interactions.
 */
class ChatController extends ControllerBase {

  /**
   * The AI Assistant runner service.
   *
   * @var mixed
   */
  protected $runner;

  /**
   * Creates an instance of the controller.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   *
   * @return static
   *   The constructed controller.
   */
  public static function create(ContainerInterface $container): static {
    $instance = new static();
    $instance->runner = $container->get('ai_assistant_api.runner');
    return $instance;
  }

  /**
   * AJAX endpoint to send a message to the configured assistant.
   */
  public function send(Request $request): JsonResponse {
    $config = $this->config('ai_chat.settings');
    $role_assistants = $config->get('role_assistants') ?: [];

    if (empty($role_assistants)) {
      return new JsonResponse(['ok' => FALSE, 'error' => 'Not configured.'], 400);
    }

    $account = $this->currentUser();
    $user_roles = $account->getRoles();

    // Collect all assistants available to the user based on their roles.
    $available_assistants = [];
    foreach ($user_roles as $role_id) {
      if (isset($role_assistants[$role_id]['assistants'])) {
        $available_assistants = array_merge($available_assistants, $role_assistants[$role_id]['assistants']);
      }
    }

    // If user has admin permission, include all configured assistants.
    if ($account->hasPermission('administer site configuration')) {
      foreach ($role_assistants as $role_config) {
        if (!empty($role_config['assistants'])) {
          $available_assistants = array_merge($available_assistants, $role_config['assistants']);
        }
      }
    }

    // Remove duplicates.
    $available_assistants = array_unique($available_assistants);

    if (empty($available_assistants)) {
      return new JsonResponse(['ok' => FALSE, 'error' => 'Access denied.'], 403);
    }

    $data = json_decode($request->getContent() ?: '[]', TRUE) ?: [];
    $message = trim((string) ($data['message'] ?? ''));
    $thread_id = isset($data['thread_id']) ? (string) $data['thread_id'] : '';
    $selected_assistant = isset($data['assistant_id']) ? (string) $data['assistant_id'] : '';

    if ($message === '') {
      return new JsonResponse(['ok' => FALSE, 'error' => 'Empty message.'], 400);
    }

    // Use selected assistant if provided and available, else use first.
    $assistant_id = $selected_assistant && in_array($selected_assistant, $available_assistants)
      ? $selected_assistant
      : reset($available_assistants);

    $assistant = $this->entityTypeManager()->getStorage('ai_assistant')->load($assistant_id);
    if (!$assistant) {
      return new JsonResponse(['ok' => FALSE, 'error' => 'Assistant not found.'], 400);
    }

    try {
      $this->runner->setAssistant($assistant);
      if (!empty($thread_id)) {
        $this->runner->setThreadsKey($thread_id);
      }
      $this->runner->setUserMessage(new UserMessage($message));
      $output = $this->runner->process();

      $normalized = $output->getNormalized();
      if (is_object($normalized) && method_exists($normalized, 'getText')) {
        $reply = (string) $normalized->getText();
      }
      else {
        $reply = (string) $normalized;
      }
      $current_thread = $this->runner->getThreadsKey();

      return new JsonResponse([
        'ok' => TRUE,
        'reply' => $reply,
        'thread_id' => $current_thread,
      ]);
    }
    catch (\Throwable $e) {
      return new JsonResponse([
        'ok' => FALSE,
        'error' => $e->getMessage(),
      ], 500);
    }
  }

}
