/**
 * @file
 * Drupal AI Console JavaScript
 * Adapted from Mautic AI Console
 */

(function (Drupal, drupalSettings) {
  'use strict';

  Drupal.behaviors.aiConsole = {
    overlay: null,
    input: null,
    output: null,
    form: null,
    isLoading: false,
    historyLoaded: false,
    consoleUrl: '/ai-console/send',
    historyUrl: '/ai-console/history',

    attach: function (context, settings) {
      // Initialize console only once
      if (context !== document) {
        return;
      }
      console.log('ai_console');

      const consoleSettings = drupalSettings.aiConsole || {};
      const keyboardShortcut = consoleSettings.keyboardShortcut || '§, `';

      // Inject console HTML if it doesn't exist
      if (!document.getElementById('ai-console-overlay')) {
        this.injectConsoleHTML();
      }

      // Bind events
      this.bindEvents(keyboardShortcut);

      // Check for URL parameter
      this.checkUrlParameter();
    },

    injectConsoleHTML: function () {
      const consoleSettings = drupalSettings.aiConsole || {};
      const placeholderText = consoleSettings.placeholderText || 'Ask your question here';

      const consoleHTML = `
        <div id="ai-console-overlay" class="ai-console-overlay" style="display: none;">
          <div class="ai-console-container">
            <div class="ai-console-header">
              <div class="ai-console-title">
                ✨ Drupal AI Console
              </div>
              <button type="button" class="ai-console-close" id="ai-console-close">
                &times;
              </button>
            </div>
            <div class="ai-console-content">
              <div class="ai-console-output" id="ai-console-output">
                <div class="ai-console-welcome">
                  Welcome to Drupal AI Console ✨<br>
                  Type your question to get started
                </div>
              </div>
              <form class="ai-console-input-form" id="ai-console-form">
                <div class="ai-console-input-container">
                  <div class="ai-console-input-wrapper">
                    <textarea
                      id="ai-console-input"
                      class="ai-console-input"
                      placeholder="${placeholderText}"
                      rows="1"></textarea>
                  </div>
                  <button type="submit" class="ai-console-submit" id="ai-console-submit">
                    ➤
                  </button>
                </div>
              </form>
            </div>
          </div>
        </div>
      `;

      document.body.insertAdjacentHTML('beforeend', consoleHTML);

      // Get references to DOM elements
      this.overlay = document.getElementById('ai-console-overlay');
      this.input = document.getElementById('ai-console-input');
      this.output = document.getElementById('ai-console-output');
      this.form = document.getElementById('ai-console-form');
    },

    bindEvents: function (keyboardShortcut) {
      const self = this;

      // Close button click
      document.addEventListener('click', function (e) {
        if (e.target.closest('.ai-console-close')) {
          self.hide();
        }
      });

      // Click outside to close
      document.addEventListener('click', function (e) {
        if (e.target === self.overlay) {
          self.hide();
        }
      });

      // Escape key to close
      document.addEventListener('keydown', function (e) {
        if (e.key === 'Escape' && self.isVisible()) {
          self.hide();
        }
      });

      // Keyboard shortcut to toggle console
      this.setupKeyboardListener(keyboardShortcut);

      // Form submission
      if (this.form) {
        this.form.addEventListener('submit', function (e) {
          e.preventDefault();
          self.processInput();
        });
      }

      // Auto-resize textarea
      if (this.input) {
        this.input.addEventListener('input', function () {
          self.autoResizeTextarea();
        });

        // Enter to submit (Shift+Enter for new line)
        this.input.addEventListener('keydown', function (e) {
          if (e.key === 'Enter' && !e.shiftKey) {
            e.preventDefault();
            if (!self.isLoading) {
              self.processInput();
            }
          }
        });
      }
    },

    setupKeyboardListener: function (shortcut) {
      const self = this;

      // Split by comma to support multiple shortcuts
      const shortcuts = shortcut.split(',').map(s => s.trim());

      // Parse each shortcut
      const parsedShortcuts = shortcuts.map(function(singleShortcut) {
        const parts = singleShortcut.split('+').map(s => s.trim());
        const modifiersLower = parts.map(p => p.toLowerCase());
        const modifiers = {
          ctrl: modifiersLower.includes('ctrl'),
          alt: modifiersLower.includes('alt'),
          shift: modifiersLower.includes('shift'),
          meta: modifiersLower.includes('meta') || modifiersLower.includes('cmd')
        };

        // Get the actual key - keep original case for special characters
        const key = parts.length > 1 ? parts[parts.length - 1] : singleShortcut.trim();

        return { modifiers: modifiers, key: key };
      });

      // Debug: log configured shortcuts
      console.log('AI Console shortcuts configured:', parsedShortcuts);

      document.addEventListener('keydown', function (e) {
        // Debug logging - log key presses to help diagnose issues
        console.log('Key pressed:', e.key, 'Code:', e.code, 'Ctrl:', e.ctrlKey, 'Alt:', e.altKey, 'Shift:', e.shiftKey, 'Meta:', e.metaKey);

        // Check if any of the shortcuts match
        const matches = parsedShortcuts.some(function(shortcutConfig) {
          const matchesModifiers = (
            e.ctrlKey === shortcutConfig.modifiers.ctrl &&
            e.altKey === shortcutConfig.modifiers.alt &&
            e.shiftKey === shortcutConfig.modifiers.shift &&
            e.metaKey === shortcutConfig.modifiers.meta
          );

          // For special characters like §, don't use toLowerCase() as it may alter them
          let matchesKey = e.key === shortcutConfig.key ||
                            e.key.toLowerCase() === shortcutConfig.key ||
                            e.code === shortcutConfig.key ||
                            e.code.toLowerCase() === shortcutConfig.key.toLowerCase();

          // Special handling for backtick - on some keyboards it's a dead key
          if (shortcutConfig.key === '`' && e.code === 'Backquote') {
            matchesKey = true;
          }

          return matchesModifiers && matchesKey;
        });

        if (matches) {
          // Don't trigger if user is typing in an input field (unless it's the console input)
          const activeElement = document.activeElement;
          if (activeElement && (activeElement.tagName === 'INPUT' || activeElement.tagName === 'TEXTAREA')) {
            if (activeElement.id !== 'ai-console-input') {
              return;
            }
          }

          e.preventDefault();
          self.toggle();
        }
      });
    },

    toggle: function () {
      if (this.isVisible()) {
        this.hide();
      } else {
        this.show();
      }
    },

    show: function () {
      if (!this.overlay) return;

      this.overlay.style.display = 'flex';
      // Small delay to trigger CSS transition
      setTimeout(() => {
        this.overlay.classList.add('show');
      }, 10);

      // Focus input
      if (this.input) {
        setTimeout(() => {
          this.input.focus();
        }, 300);
      }

      document.body.style.overflow = 'hidden';

      // Load history if not already loaded
      if (!this.historyLoaded) {
        this.loadHistory();
      }

      // Add console parameter to URL
      this.addConsoleToUrl();
    },

    hide: function () {
      if (!this.overlay) return;

      this.overlay.classList.add('slide-up');
      this.overlay.classList.remove('show');

      setTimeout(() => {
        this.overlay.style.display = 'none';
        this.overlay.classList.remove('slide-up');
      }, 300);

      document.body.style.overflow = '';

      // Remove console parameter from URL
      this.removeConsoleFromUrl();
    },

    isVisible: function () {
      return this.overlay && this.overlay.style.display !== 'none';
    },

    processInput: async function () {
      if (!this.input || !this.output || this.isLoading) return;

      const inputValue = this.input.value.trim();
      if (!inputValue) return;

      // Add user message to output
      this.addMessage(inputValue, 'user', true);

      // Clear input and show loading
      this.input.value = '';
      this.autoResizeTextarea();
      this.setLoading(true);

      // Add AI response message with thinking indicator
      const consoleSettings = drupalSettings.aiConsole || {};
      const loadingMessages = consoleSettings.loadingMessages || ['AI is thinking...'];
      const randomMessage = loadingMessages[Math.floor(Math.random() * loadingMessages.length)];
      const aiMessageDiv = this.addMessage(randomMessage, 'ai thinking', true);

      try {
        // Send request to backend
        const response = await fetch(this.consoleUrl, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'
          },
          body: JSON.stringify({
            message: inputValue,
            agent: consoleSettings.agent
          })
        });

        if (!response.ok) {
          throw new Error('Failed to communicate with AI service');
        }

        const data = await response.json();

        if (data.response) {
          aiMessageDiv.innerHTML = data.response;
          aiMessageDiv.className = aiMessageDiv.className.replace('thinking', '');
          this.output.scrollTop = this.output.scrollHeight;
        } else if (data.error) {
          aiMessageDiv.innerHTML = 'Error: ' + this.escapeHtml(data.error);
          aiMessageDiv.className = aiMessageDiv.className.replace('ai thinking', 'error');
        } else {
          throw new Error('No response received');
        }

        this.setLoading(false);

        // Restore focus to input
        if (this.input) {
          this.input.focus();
        }

      } catch (error) {
        this.setLoading(false);
        if (aiMessageDiv) {
          aiMessageDiv.innerHTML = 'Error: ' + this.escapeHtml(error.message);
          aiMessageDiv.className = aiMessageDiv.className.replace('ai thinking', 'error');
        } else {
          this.addMessage('Error: ' + this.escapeHtml(error.message), 'error', true);
        }

        // Restore focus to input
        if (this.input) {
          this.input.focus();
        }
      }
    },

    addMessage: function (text, type, animate) {
      if (!this.output) return null;

      const messageDiv = document.createElement('div');
      messageDiv.className = 'ai-console-message ' + type + (animate ? ' new' : '');

      if (type === 'ai' && !text.includes('<p>')) {
        messageDiv.innerHTML = '<p>' + text + '</p>';
      } else {
        messageDiv.innerHTML = text;
      }

      this.output.appendChild(messageDiv);
      this.output.scrollTop = this.output.scrollHeight;

      // Remove 'new' class after animation
      if (animate) {
        setTimeout(function () {
          messageDiv.classList.remove('new');
        }, 300);
      }

      return messageDiv;
    },

    setLoading: function (loading) {
      this.isLoading = loading;
      const submitBtn = document.getElementById('ai-console-submit');
      if (submitBtn) {
        submitBtn.disabled = loading;
      }
      if (this.input) {
        this.input.disabled = loading;
      }
    },

    autoResizeTextarea: function () {
      if (!this.input) return;

      this.input.style.height = 'auto';
      this.input.style.height = Math.min(this.input.scrollHeight, 120) + 'px';
    },

    checkUrlParameter: function () {
      const urlParams = new URLSearchParams(window.location.search);
      if (urlParams.get('console') === 'true') {
        this.show();
      }
    },

    addConsoleToUrl: function () {
      const url = new URL(window.location);
      url.searchParams.set('console', 'true');
      window.history.pushState({}, '', url);
    },

    removeConsoleFromUrl: function () {
      const url = new URL(window.location);
      url.searchParams.delete('console');
      window.history.pushState({}, '', url);
    },

    escapeHtml: function (text) {
      const div = document.createElement('div');
      div.textContent = text;
      return div.innerHTML;
    },

    loadHistory: async function () {
      if (!this.output) return;

      try {
        // Show loading message
        const welcomeMsg = this.output.querySelector('.ai-console-welcome');
        if (welcomeMsg) {
          welcomeMsg.innerHTML = 'Welcome to Drupal AI Console ✨<br><span style="opacity: 0.7;">Loading conversation history...</span>';
        }

        // Fetch history
        const response = await fetch(this.historyUrl, {
          method: 'GET',
          headers: {
            'X-Requested-With': 'XMLHttpRequest'
          }
        });

        if (!response.ok) {
          throw new Error('Failed to load history');
        }

        const data = await response.json();

        if (data.success && data.messages && data.messages.length > 0) {
          // Clear welcome message
          if (welcomeMsg) {
            welcomeMsg.remove();
          }

          // Add historical messages
          data.messages.forEach(entry => {
            // Add user message
            this.addMessage(entry.prompt, 'user', false);
            // Add AI response
            this.addMessage(entry.output, 'ai', false);
          });

          // Scroll to bottom
          this.output.scrollTop = this.output.scrollHeight;
        } else {
          // No history - update welcome message
          if (welcomeMsg) {
            welcomeMsg.innerHTML = 'Welcome to Drupal AI Console ✨<br>Type your question to get started';
          }
        }

        this.historyLoaded = true;
      } catch (error) {
        console.error('Error loading history:', error);
        // Update welcome message on error
        const welcomeMsg = this.output.querySelector('.ai-console-welcome');
        if (welcomeMsg) {
          welcomeMsg.innerHTML = 'Welcome to Drupal AI Console ✨<br>Type your question to get started';
        }
        this.historyLoaded = true;
      }
    }
  };

})(Drupal, drupalSettings);
