<?php

declare(strict_types=1);

namespace Drupal\ai_console\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Plugin\CachedDiscoveryClearerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\ai\AiProviderPluginManager;
use Drupal\ai_agents\PluginManager\AiAgentManager;
use Drupal\ai_agents\Task\Task;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Returns responses for AI Console routes.
 */
class AiConsoleController extends ControllerBase {

  /**
   * The AI Agent Manager.
   *
   * @var \Drupal\ai_agents\PluginManager\AiAgentManager
   */
  protected $aiAgentManager;

  /**
   * The AI Provider Plugin Manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $aiProviderPluginManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * The plugin cache clearer service.
   *
   * @var \Drupal\Core\Plugin\CachedDiscoveryClearerInterface
   */
  protected $pluginCacheClearer;

  /**
   * Constructs a new AiConsoleController object.
   *
   * @param \Drupal\ai_agents\PluginManager\AiAgentManager $ai_agent_manager
   *   The AI Agent Manager.
   * @param \Drupal\ai\AiProviderPluginManager $ai_provider_plugin_manager
   *   The AI Provider Plugin Manager.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Plugin\CachedDiscoveryClearerInterface $plugin_cache_clearer
   *   The plugin cache clearer service.
   */
  public function __construct(AiAgentManager $ai_agent_manager, AiProviderPluginManager $ai_provider_plugin_manager, AccountProxyInterface $current_user, Connection $database, TimeInterface $time, CachedDiscoveryClearerInterface $plugin_cache_clearer) {
    $this->aiAgentManager = $ai_agent_manager;
    $this->aiProviderPluginManager = $ai_provider_plugin_manager;
    $this->currentUser = $current_user;
    $this->database = $database;
    $this->time = $time;
    $this->pluginCacheClearer = $plugin_cache_clearer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.ai_agents'),
      $container->get('ai.provider'),
      $container->get('current_user'),
      $container->get('database'),
      $container->get('datetime.time'),
      $container->get('plugin.cache_clearer')
    );
  }

  /**
   * Returns the conversation history for the current user.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response with the last 100 messages.
   */
  public function history(): JsonResponse {
    try {
      // Get the last 100 messages for the current user, ordered
      // chronologically.
      $query = $this->database->select('ai_console_log', 'acl')
        ->fields('acl', [
          'id',
          'timestamp',
          'provider',
          'ai_agent',
          'model',
          'prompt',
          'output',
        ])
        ->condition('uid', $this->currentUser->id())
        ->orderBy('timestamp', 'DESC')
        ->orderBy('id', 'DESC')
        ->range(0, 100);

      $results = $query->execute()->fetchAll(\PDO::FETCH_ASSOC);

      // Reverse to get chronological order (oldest first).
      $results = array_reverse($results);

      return new JsonResponse([
        'success' => TRUE,
        'count' => count($results),
        'messages' => $results,
      ]);
    }
    catch (\Exception $e) {
      $this->getLogger('ai_console')->error('Error loading console history: @error', [
        '@error' => $e->getMessage(),
      ]);

      return new JsonResponse([
        'success' => FALSE,
        'error' => 'Failed to load conversation history.',
      ], 500);
    }
  }

  /**
   * Handles sending messages to the AI assistant.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function send(Request $request): JsonResponse {
    try {
      // Get the request data.
      $data = json_decode($request->getContent(), TRUE);

      if (!isset($data['message']) || empty($data['message'])) {
        return new JsonResponse([
          'error' => 'Message is required.',
        ], 400);
      }

      // Get the agent ID from settings or request.
      $agent_id = $data['agent'] ?? $this->config('ai_console.settings')->get('agent');

      if (empty($agent_id)) {
        return new JsonResponse([
          'error' => 'No AI agent configured. Please configure the AI Console settings.',
        ], 400);
      }
      // Load the agent config entity.
      $agent_entity = $this->entityTypeManager()->getStorage('ai_agent')->load($agent_id);

      if (!$agent_entity) {
        return new JsonResponse([
          'error' => 'The configured AI agent could not be found.',
          'agent_id' => $agent_id,
        ], 404);
      }

      // Check if the agent plugin exists in the plugin manager.
      $plugin_definitions = $this->aiAgentManager->getDefinitions();
      if (!isset($plugin_definitions[$agent_id])) {
        // Plugin not found, try to rebuild cache.
        $this->pluginCacheClearer->clearCachedDefinitions();
        $plugin_definitions = $this->aiAgentManager->getDefinitions();

        if (!isset($plugin_definitions[$agent_id])) {
          return new JsonResponse([
            'error' => 'Agent plugin not found in plugin manager.',
            'agent_id' => $agent_id,
            'available_plugins' => array_keys($plugin_definitions),
          ], 500);
        }
      }

      // Create a task from the user's message.
      $task = new Task($data['message']);
      $task->setTitle('AI Console Query');

      // Create the agent plugin instance using the agent ID.
      // The AiAgentManager treats config entity IDs as plugin IDs.
      $agent_plugin = $this->aiAgentManager->createInstance($agent_id);

      // Override provider and model if configured in AI Console settings.
      $console_config = $this->config('ai_console.settings');
      $console_provider = $console_config->get('ai_provider');
      $console_model = $console_config->get('ai_model');

      if (!empty($console_provider) && !empty($console_model)) {
        // Get the provider instance and set it on the agent.
        $provider_instance = $this->aiProviderPluginManager->createInstance($console_provider);
        $agent_plugin->setAiProvider($provider_instance);
        $agent_plugin->setModelName($console_model);
      }

      // Set the task on the agent.
      $agent_plugin->setTask($task);

      // Set runner ID for logging.
      $agent_plugin->setRunnerId('ai_console_' . time());

      // Check if the agent can solve this task.
      $solvability = $agent_plugin->determineSolvability();

      // Execute the agent to get the response.
      $response_text = '';
      if ($solvability === $agent_plugin::JOB_SOLVABLE) {
        $response_text = $agent_plugin->solve();
      }
      elseif ($solvability === $agent_plugin::JOB_SHOULD_ANSWER_QUESTION) {
        $response_text = $agent_plugin->answerQuestion();
      }
      else {
        $response_text = 'The agent cannot solve this task.';
      }

      // Get provider and model info from the agent.
      $provider_id = '';
      $model_id = '';
      if (method_exists($agent_plugin, 'getAiProvider')) {
        $provider = $agent_plugin->getAiProvider();
        if ($provider) {
          $provider_id = method_exists($provider, 'getPluginId') ? $provider->getPluginId() : '';
        }
      }
      if (method_exists($agent_plugin, 'getModelName')) {
        $model_id = $agent_plugin->getModelName() ?? '';
      }

      // Log the interaction to the database.
      $this->logInteraction(
        $data['message'],
        $response_text ?: 'No response received from the AI agent.',
        $agent_id,
        $provider_id,
        $model_id
      );

      return new JsonResponse([
        'response' => $response_text ?: 'No response received from the AI agent.',
      ]);
    }
    catch (\Exception $e) {
      // Log the error.
      $this->getLogger('ai_console')->error('Error processing AI console message: @error', [
        '@error' => $e->getMessage(),
      ]);

      return new JsonResponse([
        'error' => 'An error occurred while processing your message. Please try again.',
        'details' => $e->getMessage(),
      ], 500);
    }
  }

  /**
   * Logs an AI console interaction to the database.
   *
   * @param string $prompt
   *   The user's prompt.
   * @param string $output
   *   The AI's response.
   * @param string $agent_id
   *   The AI agent ID.
   * @param string $provider
   *   The AI provider.
   * @param string $model
   *   The AI model.
   */
  protected function logInteraction(string $prompt, string $output, string $agent_id, string $provider, string $model): void {
    try {
      $this->database->insert('ai_console_log')
        ->fields([
          'uid' => $this->currentUser->id(),
          'timestamp' => $this->time->getRequestTime(),
          'provider' => $provider,
          'ai_agent' => $agent_id,
          'model' => $model,
          'prompt' => $prompt,
          'output' => $output,
        ])
        ->execute();
    }
    catch (\Exception $e) {
      // Log the error but don't fail the request.
      $this->getLogger('ai_console')->error('Failed to log AI console interaction: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
  }

}
