<?php

declare(strict_types=1);

namespace Drupal\ai_console\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\ai\AiProviderPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure AI Console settings.
 */
class AiConsoleSettingsForm extends ConfigFormBase {

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The AI Provider Plugin Manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $aiProviderPluginManager;

  /**
   * Constructs a new AiConsoleSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\ai\AiProviderPluginManager $ai_provider_plugin_manager
   *   The AI Provider Plugin Manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, ModuleHandlerInterface $module_handler, EntityTypeManagerInterface $entity_type_manager, AiProviderPluginManager $ai_provider_plugin_manager) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->moduleHandler = $module_handler;
    $this->entityTypeManager = $entity_type_manager;
    $this->aiProviderPluginManager = $ai_provider_plugin_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('module_handler'),
      $container->get('entity_type.manager'),
      $container->get('ai.provider')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ai_console_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['ai_console.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('ai_console.settings');

    // Add description at the top.
    $form['description'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('AI Console is a quick-access chat window that appears when you press a keyboard shortcut. Think of it like the console in Minecraft or video games. You can ask questions and get instant answers without leaving your current page. Perfect for getting help while you work!') . '</p>' .
      '<p>' . $this->t("The way to configure this is the following: <br> First you create an agent with instructions. I've prepared some example instructions in the file example_instructions.md in this module. Then you enable the tools you want your agent to have access to. You don't need tools per se, you could have just instructions in the agent.") . '</p>',
    ];

    // Page visibility settings.
    $form['ux'] = [
      '#type' => 'details',
      '#title' => $this->t('Console options'),
      '#open' => TRUE,
      '#tree' => FALSE,
    ];

    // Keyboard shortcut.
    $form['ux']['keyboard_shortcut'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Keyboard shortcut to trigger AI console'),
      '#description' => $this->t('Enter one or more keyboard shortcuts to trigger the console. Use commas to separate multiple shortcuts. Examples: "§, `", "~", "Ctrl+Space", "Alt+A". Use modifier keys like Ctrl, Alt, Shift followed by a key.'),
      '#default_value' => $config->get('keyboard_shortcut') ?: '§, `',
      '#required' => TRUE,
      '#maxlength' => 100,
    ];

    // Placeholder text.
    $form['ux']['placeholder_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Placeholder text'),
      '#description' => $this->t('The placeholder text shown in the input field.'),
      '#default_value' => $config->get('placeholder_text') ?: 'Ask your question here',
      '#required' => TRUE,
      '#maxlength' => 255,
    ];

    $form['ai'] = [
      '#type' => 'details',
      '#title' => $this->t('AI options'),
      '#open' => TRUE,
      '#tree' => FALSE,
    ];

    // Load AI agents.
    $agents = [];
    if ($this->moduleHandler->moduleExists('ai_agents')) {
      $agent_storage = $this->entityTypeManager->getStorage('ai_agent');
      $agent_entities = $agent_storage->loadMultiple();
      foreach ($agent_entities as $agent) {
        $agents[$agent->id()] = $agent->label();
      }
    }

    // Agent selection.
    $form['ai']['agent'] = [
      '#type' => 'select',
      '#title' => $this->t('AI Agent'),
      '#description' => $this->t('Select the AI agent to use for the console. You can create or configure agents at <a href="/admin/structure/ai-agent">AI Agents</a>.'),
      '#options' => $agents,
      '#empty_option' => $this->t('- Select an agent -'),
      '#default_value' => $config->get('agent'),
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::agentLinkCallback',
        'wrapper' => 'agent-link-wrapper',
        'event' => 'change',
      ],
    ];

    // Agent configuration link.
    $form['ai']['agent_link'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'agent-link-wrapper'],
    ];

    $selected_agent = $form_state->getValue('agent') ?? $config->get('agent');
    if (!empty($selected_agent)) {
      $form['ai']['agent_link']['markup'] = [
        '#type' => 'markup',
        '#markup' => '<p>' . $this->t('➜ <a href="@url">Configure this agent</a>', [
          '@url' => '/admin/structure/ai-agent/' . $selected_agent,
        ]) . '</p>',
      ];
    }

    // AI Provider and Model selection - combined dropdown.
    $model_options = $this->getAvailableModels();
    $default_value = '';
    if ($config->get('ai_provider') && $config->get('ai_model')) {
      $default_value = $config->get('ai_provider') . '::' . $config->get('ai_model');
    }

    $form['ai']['ai_model_combined'] = [
      '#type' => 'select',
      '#title' => $this->t('AI Model'),
      '#description' => $this->t('Select the AI provider and model to use for the console.'),
      '#options' => $model_options,
      '#default_value' => $default_value,
      '#empty_option' => $this->t('- Select a model -'),
      '#required' => FALSE,
      '#weight' => -10,
    ];

    // Loading messages.
    $default_messages = $config->get('loading_messages') ?: [
      '🧠 AI is thinking...',
      '✨ Making magic happen...',
      '📡 Querying the knowledge of your incredible CMS...',
      '🧱 Crafting your answer...',
    ];

    $form['ai']['loading_messages'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Loading messages'),
      '#description' => $this->t('Enter loading messages to display while waiting for AI response. One message per line. A random message will be chosen each time.'),
      '#default_value' => is_array($default_messages) ? implode("\n", $default_messages) : $default_messages,
      '#rows' => 5,
      '#required' => TRUE,
    ];

    // History retention settings.
    $form['history_retention'] = [
      '#type' => 'details',
      '#title' => $this->t('Log retention'),
      '#open' => TRUE,
      '#tree' => FALSE,
    ];

    $form['history_retention']['retention_days'] = [
      '#type' => 'number',
      '#title' => $this->t('Days to keep history'),
      '#description' => $this->t('Number of days to keep AI console conversation history. Older entries will be automatically deleted by cron. Set to 0 to keep history forever.'),
      '#default_value' => $config->get('retention_days') ?: 30,
      '#min' => 0,
      '#max' => 3650,
      '#required' => TRUE,
    ];

    // Page visibility settings.
    $form['visibility'] = [
      '#type' => 'details',
      '#title' => $this->t('Page visibility'),
      '#open' => TRUE,
      '#tree' => FALSE,
    ];

    $form['visibility']['visibility_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Show console on specific pages'),
      '#options' => [
        'show' => $this->t('Show for the listed pages'),
        'hide' => $this->t('Hide for the listed pages'),
      ],
      '#default_value' => $config->get('visibility_mode') ?: 'show',
    ];

    $form['visibility']['pages'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Pages'),
      '#description' => $this->t("Specify pages by using their paths. Enter one path per line. The '*' character is a wildcard. An example path is %user-wildcard for every user page. %front is the front page.", [
        '%user-wildcard' => '/user/*',
        '%front' => '<front>',
      ]),
      '#default_value' => $config->get('pages') ?: '',
      '#rows' => 10,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * AJAX callback for agent selection change.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The agent link element.
   */
  public function agentLinkCallback(array &$form, FormStateInterface $form_state): array {
    $selected_agent = $form_state->getValue('agent');

    $element = [
      '#type' => 'container',
      '#attributes' => ['id' => 'agent-link-wrapper'],
    ];

    if (!empty($selected_agent)) {
      $element['markup'] = [
        '#type' => 'markup',
        '#markup' => '<p>' . $this->t('➜ <a href="@url" target="_blank">Configure this agent</a>', [
          '@url' => '/admin/structure/ai-agent/' . $selected_agent,
        ]) . '</p>',
      ];
    }

    return $element;
  }

  /**
   * Get available models from all providers.
   *
   * @return array
   *   Array of model options grouped by provider.
   */
  protected function getAvailableModels(): array {
    $options = [];

    // Get all provider definitions.
    $providers = $this->aiProviderPluginManager->getDefinitions();

    foreach ($providers as $provider_id => $provider_definition) {
      try {
        // Create provider instance.
        $provider = $this->aiProviderPluginManager->createInstance($provider_id);

        // Get models for chat operation.
        $models = $provider->getConfiguredModels('chat');

        if (!empty($models)) {
          // Group models by provider.
          foreach ($models as $model_id => $model_label) {
            $options[$provider_definition['label']->__toString()][$provider_id . '::' . $model_id] = $model_label;
          }
        }
      }
      catch (\Exception $e) {
        // Skip providers that fail to instantiate.
        continue;
      }
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Parse the combined provider::model value.
    $combined_value = $form_state->getValue('ai_model_combined');
    $ai_provider = '';
    $ai_model = '';

    if (!empty($combined_value) && strpos($combined_value, '::') !== FALSE) {
      [$ai_provider, $ai_model] = explode('::', $combined_value, 2);
    }

    // Parse loading messages - split by newline and filter empty lines.
    $loading_messages_text = $form_state->getValue('loading_messages');
    $loading_messages = array_filter(
      array_map('trim', explode("\n", $loading_messages_text)),
      function ($message) {
        return !empty($message);
      }
    );

    $this->config('ai_console.settings')
      ->set('ai_provider', $ai_provider)
      ->set('ai_model', $ai_model)
      ->set('agent', $form_state->getValue('agent'))
      ->set('keyboard_shortcut', $form_state->getValue('keyboard_shortcut'))
      ->set('placeholder_text', $form_state->getValue('placeholder_text'))
      ->set('loading_messages', array_values($loading_messages))
      ->set('retention_days', $form_state->getValue('retention_days'))
      ->set('visibility_mode', $form_state->getValue('visibility_mode'))
      ->set('pages', $form_state->getValue('pages'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
