<?php

namespace Drupal\ai_content_advisor;

use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Access\AccessResult;

/**
 * Access controller for the AI Content Advisor Report entity.
 *
 * @see \Drupal\ai_content_advisor\Entity\AiContentAdvisorReport.
 */
class AiContentAdvisorReportAccessControlHandler extends EntityAccessControlHandler {

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    /** @var \Drupal\ai_content_advisor\Entity\AiContentAdvisorReport $entity */
    
    switch ($operation) {
      case 'view':
        // Users can view reports if they have permission to view reports
        // or if they have admin permission
        return AccessResult::allowedIfHasPermissions($account, [
          'view ai content advisor reports',
          'administer ai content advisor',
        ], 'OR');

      case 'create':
        // Only users with create permission can create reports
        return AccessResult::allowedIfHasPermission($account, 'create ai content advisor reports');

      case 'update':
        // Users can edit reports if they:
        // - Have admin permission, OR
        // - Have create permission AND are the author of the report
        $admin_access = AccessResult::allowedIfHasPermission($account, 'administer ai content advisor');
        
        if ($admin_access->isAllowed()) {
          return $admin_access;
        }
        
        $create_permission = AccessResult::allowedIfHasPermission($account, 'create ai content advisor reports');
        $is_owner = AccessResult::allowedIf($account->id() == $entity->getOwnerId());
        
        return $create_permission->andIf($is_owner);

      case 'delete':
        // Users can delete reports if they:
        // - Have admin permission, OR
        // - Have create permission AND are the author of the report
        $admin_access = AccessResult::allowedIfHasPermission($account, 'administer ai content advisor');
        
        if ($admin_access->isAllowed()) {
          return $admin_access;
        }
        
        $create_permission = AccessResult::allowedIfHasPermission($account, 'create ai content advisor reports');
        $is_owner = AccessResult::allowedIf($account->id() == $entity->getOwnerId());
        
        return $create_permission->andIf($is_owner);
    }

    // Unknown operation, no opinion.
    return AccessResult::neutral();
  }

  /**
   * {@inheritdoc}
   */
  protected function checkCreateAccess(AccountInterface $account, array $context, $entity_bundle = NULL) {
    return AccessResult::allowedIfHasPermission($account, 'create ai content advisor reports');
  }

}