<?php

namespace Drupal\ai_content_advisor\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\user\UserInterface;

/**
 * Defines the AI Content Advisor Report entity.
 *
 * @ContentEntityType(
 *   id = "ai_content_advisor_report",
 *   label = @Translation("AI Content Advisor Report"),
 *   label_collection = @Translation("AI Content Advisor Reports"),
 *   label_singular = @Translation("report"),
 *   label_plural = @Translation("reports"),
 *   label_count = @PluralTranslation(
 *     singular = "@count report",
 *     plural = "@count reports"
 *   ),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\ai_content_advisor\AiContentAdvisorReportListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "form" = {
 *       "default" = "Drupal\Core\Entity\ContentEntityForm",
 *       "add" = "Drupal\Core\Entity\ContentEntityForm",
 *       "edit" = "Drupal\Core\Entity\ContentEntityForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *     },
 *     "access" = "Drupal\ai_content_advisor\AiContentAdvisorReportAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "ai_content_advisor_report",
 *   admin_permission = "administer ai content advisor",
 *   entity_keys = {
 *     "id" = "id",
 *     "uuid" = "uuid",
 *     "uid" = "uid",
 *   },
 *   links = {
 *     "canonical" = "/admin/reports/ai-content-advisor/{ai_content_advisor_report}",
 *     "add-form" = "/admin/reports/ai-content-advisor/add",
 *     "edit-form" = "/admin/reports/ai-content-advisor/{ai_content_advisor_report}/edit",
 *     "delete-form" = "/admin/reports/ai-content-advisor/{ai_content_advisor_report}/delete",
 *     "collection" = "/admin/reports/ai-content-advisor",
 *   },
 *   field_ui_base_route = "ai_content_advisor_report.settings"
 * )
 */
class AiContentAdvisorReport extends ContentEntityBase {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    // Entity type ID field.
    $fields['entity_type_id'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Entity type ID'))
      ->setDescription(t('The entity type ID of the analyzed content.'))
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 1,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // Entity ID field.
    $fields['entity_id'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Entity ID'))
      ->setDescription(t('The entity ID of the analyzed content.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_integer',
        'weight' => 2,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // Revision ID field.
    $fields['revision_id'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Revision ID'))
      ->setDescription(t('The revision ID of the analyzed content.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'number_integer',
        'weight' => 3,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // Language code field.
    $fields['langcode'] = BaseFieldDefinition::create('language')
      ->setLabel(t('Language'))
      ->setDescription(t('The language code of the analyzed content.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'language',
        'weight' => 4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // URL field.
    $fields['url'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('URL'))
      ->setDescription(t('The URL of the analyzed content.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // User ID field (author).
    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Author'))
      ->setDescription(t('The user who created the report.'))
      ->setRevisionable(FALSE)
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'author',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 5,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDefaultValueCallback('Drupal\ai_content_advisor\Entity\AiContentAdvisorReport::getCurrentUserId');

    // Report field.
    $fields['report'] = BaseFieldDefinition::create('text_long')
      ->setLabel(t('Report'))
      ->setDescription(t('The AI analysis report content.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'text_default',
        'weight' => 6,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // Report type field.
    $fields['report_type'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Report Type'))
      ->setDescription(t('The type of report generated.'))
      ->setRequired(TRUE)
      ->setDefaultValue('full')
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 7,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // Prompt field.
    $fields['prompt'] = BaseFieldDefinition::create('text_long')
      ->setLabel(t('Prompt'))
      ->setDescription(t('The prompt used to generate the report.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'text_default',
        'weight' => 8,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // HTML analyzed field.
    $fields['html_analyzed'] = BaseFieldDefinition::create('text_long')
      ->setLabel(t('HTML Analyzed'))
      ->setDescription(t('The HTML content that was analyzed.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'text_default',
        'weight' => 9,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // Timestamp field.
    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the report was created.'));

    // Feedback quality rating field.
    $fields['feedback_quality_rating'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Quality Rating'))
      ->setDescription(t('Quality rating of the recommendations.'))
      ->setSettings([
        'allowed_values' => [
          'very_low' => 'Very Low',
          'low' => 'Low',
          'fair' => 'Fair',
          'good' => 'Good',
          'very_good' => 'Very Good',
        ],
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'list_default',
        'weight' => 10,
      ])
      ->setDisplayOptions('form', [
        'type' => 'options_buttons',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // Feedback comments field.
    $fields['feedback_comments'] = BaseFieldDefinition::create('text_long')
      ->setLabel(t('Feedback Comments'))
      ->setDescription(t('Comments about what was not accurate or useful in the recommendations.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'text_default',
        'weight' => 11,
      ])
      ->setDisplayOptions('form', [
        'type' => 'text_textarea',
        'weight' => 11,
        'settings' => [
          'rows' => 4,
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // Feedback author name field.
    $fields['feedback_author_name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Feedback Author Name'))
      ->setDescription(t('Name of the person providing feedback.'))
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 12,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 12,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // Feedback timestamp field.
    $fields['feedback_timestamp'] = BaseFieldDefinition::create('timestamp')
      ->setLabel(t('Feedback Timestamp'))
      ->setDescription(t('The time when feedback was provided.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 13,
      ])
      ->setDisplayConfigurable('view', TRUE);

    return $fields;
  }

  /**
   * Default value callback for 'uid' base field definition.
   *
   * @see ::baseFieldDefinitions()
   *
   * @return array
   *   An array of default values.
   */
  public static function getCurrentUserId() {
    return [\Drupal::currentUser()->id()];
  }

  /**
   * Gets the report author.
   *
   * @return \Drupal\user\UserInterface
   *   The report author.
   */
  public function getOwner() {
    return $this->get('uid')->entity;
  }

  /**
   * Gets the report author user ID.
   *
   * @return int
   *   The report author user ID.
   */
  public function getOwnerId() {
    return $this->get('uid')->target_id;
  }

  /**
   * Sets the report author.
   *
   * @param \Drupal\user\UserInterface $account
   *   The author user entity.
   *
   * @return $this
   */
  public function setOwner(UserInterface $account) {
    $this->set('uid', $account->id());
    return $this;
  }

  /**
   * Sets the report author by user ID.
   *
   * @param int $uid
   *   The author user id.
   *
   * @return $this
   */
  public function setOwnerId($uid) {
    $this->set('uid', $uid);
    return $this;
  }

}