<?php

namespace Drupal\ai_content_advisor\Form;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\Routing\CurrentRouteMatch;
use Drupal\Core\Session\AccountInterface;
use Drupal\node\NodeInterface;
use Drupal\ai_content_advisor\AiContentAdvisorAnalyzer;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Analyze a Content form.
 */
class AnalyzeNodeForm extends FormBase {
  use MessengerTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Current route match.
   *
   * @var \Drupal\Core\Routing\CurrentRouteMatch
   */
  protected $currentRouteMatch;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * AI analyzer.
   *
   * @var \Drupal\ai_content_advisor\AiContentAdvisorAnalyzer
   */
  protected $analyzer;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The config.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a new Weight table form object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Routing\CurrentRouteMatch $current_route_match
   *   The language manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\ai_content_advisor\AiContentAdvisorAnalyzer $analyzer
   *   AI analyzer.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   Current user.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   */
  public function __construct(
      EntityTypeManagerInterface $entity_type_manager,
      CurrentRouteMatch $current_route_match,
      LanguageManagerInterface $language_manager,
      AiContentAdvisorAnalyzer $analyzer,
      DateFormatterInterface $date_formatter,
      MessengerInterface $messenger,
      AccountInterface $current_user,
      ConfigFactoryInterface $config_factory,
      ModuleHandlerInterface $module_handler,
    ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->currentRouteMatch = $current_route_match;
    $this->languageManager = $language_manager;
    $this->analyzer = $analyzer;
    $this->dateFormatter = $date_formatter;
    $this->messenger = $messenger;
    $this->currentUser = $current_user;
    $this->config = $config_factory->get('ai_content_advisor.configuration');
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('current_route_match'),
      $container->get('language_manager'),
      $container->get('ai_content_advisor.service'),
      $container->get('date.formatter'),
      $container->get('messenger'),
      $container->get('current_user'),
      $container->get('config.factory'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'analyze_url_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $hash = NULL): array {

    // Get report types from entities.
    $report_types = $this->getReportTypes();
    $report_type_default_value = !empty($report_types) ? array_key_first($report_types) : 'full';

    $form['#id'] = 'analyze-url-form';

    $form['container'] = [
      '#type' => 'container',
      '#open' => TRUE,
      '#collapsible' => FALSE,
      '#attributes' => [
        'class' => [
          'analyze-url-form__container',
        ],
      ],
    ];

    // First make sure that the AI config is set up.
    $model_and_provider_string = $this->config->get('provider_and_model') ?? '';
    $model_and_provider = explode('__', $model_and_provider_string);

    if (count($model_and_provider) !== 2) {
      $form['container']['header'] = [
        '#type' => 'markup',
        '#markup' => $this->t('<p>Missing provider, select on at %settings.</p>', [
          '%settings' => Link::createFromRoute('AI Content Advisor module settings', 'ai_content_advisor.settings')->toString(),
        ]),
        '#attributes' => [
          'class' => [
            'form--header',
          ],
        ],
      ];
      return $form;
    }

    $entity_type_id = 'node';
    $entity = $this->currentRouteMatch->getParameter($entity_type_id);

    if (!($entity instanceof NodeInterface)) {
      $form['container']['header'] = [
        '#type' => 'markup',
        '#markup' => '
          <p>Node not found.</p>
        ',
        '#attributes' => [
          'class' => [
            'form--header',
          ],
        ],
      ];
      return $form;
    }

    $form['container']['header'] = [
      '#type' => 'markup',
      '#markup' => '
        <p>This form facilitates the generation of Content Advisor reports by allowing you to specify criteria for content analysis.
        Modify the prompt in the options below. The report generation can take some time so stay on the page while it is happening.</p>
      ',
      '#attributes' => [
        'class' => [
          'form--header',
        ],
      ],
    ];

    // Show the previous reports if there are any.
    $previous_reports = $this->analyzer->getReports($entity->id());
    ksort($previous_reports, SORT_NUMERIC);

    $latest_report = end($previous_reports);

    // Latest report.
    if (count($previous_reports) > 0) {
      // Create a details for the latest report.
      $form['container']['reports'] = [
        '#type' => 'details',
        '#title' => $this->t('Latest report'),
        '#open' => TRUE,
        '#collapsible' => FALSE,
        '#attributes' => [
          'class' => [
            'reports__container',
          ],
          'style' => 'max-width: 1200px;',
        ],
      ];

      // Show the created on datetime.
      $time_ago = $this->dateFormatter->formatTimeDiffSince($latest_report['timestamp']) . ' ' . $this->t('ago');

      // Get the revision if it's been stored.
      $revision_row = '';
      if (!empty($latest_report['revision_id'])) {
        $revision_row = $this->t('<div class="report--revision-id"><label><strong>Created for revision</strong> <a href="/node/:nid/revisions/:revision_id/view">#:revision_id</a></label></div>', [
          ':nid' => $entity->id(),
          ':revision_id' => $latest_report['revision_id'],
        ]);
      }

      $report_type_id = $latest_report['report_type'] ?? 'full';
      $report_type_default_value = $report_type_id;
      $report_type = $report_types[$report_type_id] ?? (!empty($report_types) ? reset($report_types) : 'Unknown');

      $disclaimer = $this->config->get('disclaimer_text') ?? '<h3>Notice</h3><p>These recommendations are generated by AI and may include errors. Please review carefully before making changes. If you find issues, share feedback below.</p>';
      $html_report = $this->formatAiResponseWithCode($latest_report['report']);
      $form['container']['reports']['latest'] = [
        '#type' => 'markup',
        '#children' => '
        <div class="report--timestamp"><label><strong>Report created:</strong> ' . $time_ago . '</label></div>
        <div class="report--type"><label><strong>Report type:</strong> ' . $report_type . '</label></div>' .
        $revision_row .
        '<div class="report--disclaimer">' . $disclaimer . '</div>' .
        "{$html_report}",
      ];

      $form['container']['reports']['prompt'] = [
        '#type' => 'textarea',
        '#title' => 'Prompt used',
        '#disabled' => TRUE,
        '#readonly' => TRUE,
        '#value' => $latest_report['prompt'],
      ];

      $form['container']['reports']['html_analyzed'] = $this->buildHtmlAnalyzedField(
        $latest_report['html_analyzed'] ?? ''
      );

      // Add a feedback form if feedback is enabled globally and for this report type.
      $report_type_id = $latest_report['report_type'] ?? 'full';
      $global_feedback_enabled = $this->config->get('feedback_enabled') ?? TRUE;
      $report_type_feedback_enabled = $this->isFeedbackEnabledForReportType($report_type_id);

      if ($global_feedback_enabled && $report_type_feedback_enabled) {
        $form['container']['reports']['feedback'] = $this->buildFeedbackForm($latest_report);
      }
    }

    // Remove the latest report from the list.
    if (!empty($latest_report)) {
      unset($previous_reports[$latest_report['rid']]);
    }

    $report_count = count($previous_reports);

    // Older reports.
    if ($report_count >= 1) {
      // Create a detail for the latest report.
      $form['container']['older_reports'] = [
        '#type' => 'details',
        '#title' => $this->t('Older reports'),
        '#open' => FALSE,
        '#collapsible' => FALSE,
        '#attributes' => [
          'class' => [
            'older-reports__container',
          ],
          'style' => 'max-width: 1200px;',
        ],
      ];


      $report_counter = 1;
      foreach ($previous_reports as $i => $report) {
        $report_number = $report_counter;

        $report_type_id = $report['report_type'] ?? 'full';
        $report_type = $report_types[$report_type_id] ?? (!empty($report_types) ? reset($report_types) : 'Unknown');
        // Show the created on datetime.
        $time_ago = $this->dateFormatter->formatTimeDiffSince($report['timestamp']) . ' ' . $this->t('ago');

        $date = $this->dateFormatter->format($report['timestamp'], 'custom', 'Y-m-d H:i:s');
        $form['container']['older_reports']['report_' . $i] = [
          '#type' => 'details',
          '#title' => new FormattableMarkup("Report #:report_number - :report_type. Report created: <span title=\"$date\">:time_ago</span>", [
            ':report_number' => $report_number,
            ':report_type' => $report_type,
            ':time_ago' => $time_ago,
          ]),
          '#open' => FALSE,
          '#collapsible' => FALSE,
          '#attributes' => [
            'class' => [
              'reports__container',
            ],
          ],
        ];

        // Get the revision if it's been stored.
        $revision_row = '';
        if (!empty($report['revision_id'])) {
          $revision_row = $this->t('<div class="report--revision-id"><label><strong>Created for revision</strong> <a href="/node/:nid/revisions/:revision_id/view">#:revision_id</a></label></div>', [
            ':nid' => $entity->id(),
            ':revision_id' => $report['revision_id'],
          ]);
        }

        $disclaimer = $this->config->get('disclaimer_text') ?? '<h3>Notice</h3><p>These recommendations are generated by AI and may include errors. Please review carefully before making changes. If you find issues, share feedback below.</p>';
        $form['container']['older_reports']['report_' . $i]['report'] = [
          '#type' => 'markup',
          '#children' => $revision_row . '<div class="report--disclaimer">' . $disclaimer . '</div>' . $report['report'],
        ];
        $form['container']['older_reports']['report_' . $i]['prompt'] = [
          '#type' => 'textarea',
          '#title' => 'Prompt used',
          '#disabled' => TRUE,
          '#readonly' => TRUE,
          '#value' => $report['prompt'],
        ];

        $form['container']['older_reports']['report_' . $i]['html_analyzed'] = $this->buildHtmlAnalyzedField(
          $report['html_analyzed'] ?? ''
        );

        // Add feedback form if feedback is enabled globally and for this report type
        $older_report_type_id = $report['report_type'] ?? 'full';
        $older_report_type_feedback_enabled = $this->isFeedbackEnabledForReportType($older_report_type_id);

        if ($global_feedback_enabled && $older_report_type_feedback_enabled) {
          $form['container']['older_reports']['report_' . $i]['feedback'] = $this->buildFeedbackForm($report);
        }

        $report_counter++;
      }
    }

      // Settings.
    $can_create_new_reports = $this->currentUser->hasPermission('create ai content advisor reports');
    $form['container']['new_report'] = [
       '#type' => 'details',
       '#access' => $can_create_new_reports,
       '#title' => $this->t('Create a New Content Analysis Report'),
       '#open' => FALSE,
       '#collapsible' => TRUE,
       '#attributes' => [
         'class' => [
           'analyze-url-settings__container',
         ],
       ],
     ];

    $form['container']['new_report']['report_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Report type'),
      '#options' => $report_types,
      '#default_value' => $report_type_default_value,
      '#description' => $this->t('Select the type of report to generate.'),
      '#ajax' => [
        'callback' => [$this, 'updatePrompt'],
        'wrapper' => 'analyze-url-prompt__container',
        'event' => 'change',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Updating prompt...'),
        ],
        'disable-refocus' => TRUE,
      ],
    ];

    // Prompt.
    // Get the default / custom prompt.
    $form['container']['new_report']['prompt'] = [
      '#type' => 'details',
      '#title' => $this->t('Customize Your Analysis Prompt'),
      '#open' => FALSE,
      '#collapsible' => TRUE,
      '#attributes' => [
        'class' => [
          'analyze-url-prompt__container',
        ],
        'id' => 'analyze-url-prompt__container',
        'style' => 'max-width: 1200px;',
      ],
    ];

    $form['container']['new_report']['prompt']['prompt_to_use'] = [
      '#type' => 'textarea',
      '#title' => 'Your Analysis Prompt',
      '#default_value' => $this->getPromptForReportType($form_state->getValue('report_type')),
      '#rows' => 20,
      '#description' => $this->t('Modify the analysis prompt as needed or use the default settings in the %settings.', [
        '%settings' => Link::createFromRoute('module settings', 'ai_content_advisor.settings')->toString(),
      ]),
      '#required' => TRUE,
    ];

    // If the entity is moderated, show some extra controls.
    if (!empty($entity->moderation_state->value)) {
      /** @var \Drupal\content_moderation\ModerationInformation $moderation_information_service */
      $moderationInformationService = \Drupal::service('content_moderation.moderation_information');
      $workflow = $moderationInformationService->getWorkflowForEntity($entity);
      $storage = $this->entityTypeManager->getStorage($entity_type_id);

      // Find the revisions and build the options.
      $revisions = $storage->revisionIds($entity);
      $published_revision_id = NULL;
      $revisions = array_reverse($revisions);
      $options = [];
      foreach ($revisions as $revision_id) {
        $revision = $storage->loadRevision($revision_id);
        if ($revision->isPublished() && empty($published_revision_id)) {
          $published_revision_id = $revision_id;
        }
        $created_at = $this->dateFormatter->format($revision->getChangedTime(), 'short');

        $options[$revision_id] = $this->t('#:revision_id - :revision_created - :revision_label:current_label', [
          ':revision_id' => $revision_id,
          ':revision_created' => $created_at,
          ':revision_label' => $workflow->getTypePlugin()->getState($revision->moderation_state->value)->label(),
          ':current_label' => ($revision_id === $published_revision_id) ? ' (current revision)' : '',
        ]);
      }

      $form['container']['new_report']['moderation_state'] = [
        '#type' => 'details',
        '#title' => $this->t('Select Content Revision'),
        '#open' => TRUE,
        '#collapsible' => FALSE,
        '#attributes' => [
          'class' => [
            'analyze-url-settings__container',
          ],
        ],
      ];
      $form['container']['new_report']['moderation_state']['revision_id'] = [
        '#type' => 'select',
        '#title' => 'Revision to analyze',
        '#required' => TRUE,
        '#default_value' => $published_revision_id,
        '#options' => $options,
      ];
    }

    $form['container']['new_report']['request_as_anonymous'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Analyze using anonymous visitor'),
      '#default_value' => TRUE,
      '#description' => $this->t('Check this box to analyze the page as an anonymous visitor. This will take into account any access restrictions that are in place.'),
    ];

    // Footer.
    $form['container']['new_report']['footer'] = [
      '#type' => 'markup',
      '#markup' => '
        <p>Generating the report takes some time, don\'t navigate away from this page during it.</p>
      ',
      '#attributes' => [
        'class' => [
          'form--footer',
        ],
      ],
    ];

    // Fields to store entity info to submit.
    $form['container']['entity_id'] = [
      '#type' => 'hidden',
      '#title' => 'Entity ID',
      '#required' => TRUE,
      '#value' => $entity->id(),
    ];
    $form['container']['langcode'] = [
      '#type' => 'hidden',
      '#title' => 'Entity langcode',
      '#required' => TRUE,
      '#value' => $entity->language()->getId(),
    ];

    // Form actions.
    $form['container']['new_report']['actions'] = ['#type' => 'actions'];
    $form['container']['new_report']['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Analyze'),
      '#attributes' => [
        'class' => [
          'btn--analyze',
        ],
      ],
      '#ajax' => [
        'callback' => [static::class, 'ajaxSubmit'],
        'wrapper' => 'analyze-url-form',
      ],
    ];

    return $form;
  }

  /**
   * Ajax callback to update the prompt.
   */
  public function updatePrompt(array &$form, FormStateInterface $form_state) {
    $report_type = $form_state->getValue('report_type');
    $title = $this->t('Your Analysis Prompt', []);
    $form['container']['new_report']['prompt']['#open'] = TRUE;
    $form['container']['new_report']['prompt']['prompt_to_use']['#value'] = $this->getPromptForReportType($report_type);
    return $form['container']['new_report']['prompt'];
  }

  /**
   * Get available report types from entities.
   *
   * @return array
   *   Array of report type options keyed by machine name.
   */
  private function getReportTypes() {
    $options = [];

    try {
      /** @var \Drupal\ai_content_advisor\Entity\AiContentAdvisorReportType[] $report_types */
      $report_types = $this->entityTypeManager
        ->getStorage('ai_content_advisor_report_type')
        ->loadByProperties(['status' => TRUE]);

      foreach ($report_types as $report_type) {
        $options[$report_type->id()] = $report_type->label();
      }
    } catch (\Exception $e) {
      // Fallback to hardcoded options if entity storage not available yet.
      $options = [
        'full' => $this->t('Full'),
        'topic_authority' => $this->t('Topic Authority'),
        'natural_language' => $this->t('Natural Language Use'),
        'link_analysis' => $this->t('Link Analysis'),
        'headings_and_structure' => $this->t('Headings and Structure'),
      ];
    }

    return $options;
  }

  /**
   * Helper function to get the prompt based on the report type.
   */
  private function getPromptForReportType($report_type) {
    if (empty($report_type)) {
      $report_type = 'full';
    }

    try {
      /** @var \Drupal\ai_content_advisor\Entity\AiContentAdvisorReportType $report_type_entity */
      $report_type_entity = $this->entityTypeManager
        ->getStorage('ai_content_advisor_report_type')
        ->load($report_type);

      if ($report_type_entity && $report_type_entity->status()) {
        return $report_type_entity->getPrompt();
      }
    } catch (\Exception $e) {
      // Log error if entity storage is not available.
      \Drupal::logger('ai_content_advisor')->error('Could not load AI Content Advisor report type entity: @message', ['@message' => $e->getMessage()]);
    }

    // Return empty prompt if no entity found.
    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $entity_id = $form_state->getValue('entity_id');
    $langcode = !empty($form_state->getValue('langcode')) ? $form_state->getValue('langcode') : NULL;
    $revision_id = !empty($form_state->getValue('revision_id')) ? $form_state->getValue('revision_id') : NULL;
    $prompt = $form_state->getValue('prompt_to_use') ?? '';

    // Set options.
    $options = [
      'request_as_anonymous' => $form_state->getValue('request_as_anonymous') ?? TRUE,
      'report_type' => $form_state->getValue('report_type') ?? 'full',
    ];

    // Analyze node URL using our custom analyzer service.
    $this->analyzer->analyzeEntity($prompt, 'node', $entity_id, $revision_id, 'full', $langcode, $options);
    $form_state->setRebuild();
  }

  /**
   * Fetch and parse results.
   */
  public static function ajaxSubmit(array &$form, FormStateInterface $form_state): array {
    return $form;
  }

  /**
   * Formats an AI-generated response by escaping unwanted HTML while preserving certain allowed tags.
   *
   * @param string $response
   *   The AI-generated response containing potential HTML content.
   *
   * @return string
   *   The processed response with HTML properly escaped and only the allowed tags preserved.
   */
  function formatAiResponseWithCode(string $response, array $allowedTags = ['p', 'ul', 'ol', 'li', 'blockquote', 'pre', 'code', 'strong', 'em', 'br', 'h1', 'h2', 'h3', 'h4', 'hr']) {
    $tokens = [];
    $tokenId = 0;

    // Create pattern for allowed tags
    $allowedPattern = implode('|', array_map('preg_quote', $allowedTags));

    // Replace allowed tags with tokens
    $processed = preg_replace_callback(
      '/(<\/?(?:' . $allowedPattern . ')(?:\s[^>]*)?>)/i',
      function($matches) use (&$tokens, &$tokenId) {
        $token = '__TOKEN_' . $tokenId . '__';
        $tokens[$token] = $matches[1];
        $tokenId++;
        return $token;
      },
      $response
    );

    // Escape only < and > characters, preserve quotes
    $escaped = str_replace(['<', '>'], ['&lt;', '&gt;'], $processed);

    // Also escape & but not if it's already part of an entity
    $escaped = preg_replace('/&(?!(?:[a-zA-Z][a-zA-Z0-9]*|#[0-9]+|#x[0-9a-fA-F]+);)/', '&amp;', $escaped);

    // Restore allowed tags
    foreach ($tokens as $token => $originalTag) {
      $escaped = str_replace($token, $originalTag, $escaped);
    }

    return $escaped;

  }

  /**
   * Checks if feedback is enabled for a given report type.
   *
   * @param string $report_type_id
   *   The report type machine name.
   *
   * @return bool
   *   TRUE if feedback is enabled, FALSE otherwise.
   */
  private function isFeedbackEnabledForReportType($report_type_id) {
    try {
      /** @var \Drupal\ai_content_advisor\Entity\AiContentAdvisorReportType $report_type_entity */
      $report_type_entity = $this->entityTypeManager
        ->getStorage('ai_content_advisor_report_type')
        ->load($report_type_id);

      if ($report_type_entity) {
        return $report_type_entity->isFeedbackEnabled();
      }
    } catch (\Exception $e) {
      \Drupal::logger('ai_content_advisor')->error('Could not check feedback enabled status: @message', ['@message' => $e->getMessage()]);
    }

    return FALSE;
  }

  /**
   * Builds the feedback form for a report.
   *
   * @param array $report
   *   The report data array.
   *
   * @return array
   *   The feedback form render array.
   */
  private function buildFeedbackForm(array $report): array {
    $rid = $report['rid'];
    $has_feedback = !empty($report['feedback_quality_rating']);

    $form = [
      '#type' => 'details',
      '#title' => $this->t('Report Feedback'),
      '#open' => TRUE,
      '#collapsible' => TRUE,
      '#attributes' => [
        'class' => [
          'feedback-form__container',
        ],
        'id' => 'feedback-container-' . $rid,
        'style' => 'border: 1px solid #ddd; padding: 15px; margin-top: 20px;',
      ],
    ];

    $quality_rating_options = [
      'very_low' => $this->t('Very low'),
      'low' => $this->t('Low'),
      'fair' => $this->t('Fair'),
      'good' => $this->t('Good'),
      'very_good' => $this->t('Very good'),
    ];

    if ($has_feedback) {
      $form['existing_feedback'] = [
        '#type' => 'container',
        '#attributes' => [
          'id' => 'existing-feedback-' . $rid,
        ],
      ];

      $form['existing_feedback']['display'] = [
        '#type' => 'markup',
        '#markup' => $this->buildExistingFeedbackDisplay($report),
      ];

      $form['existing_feedback']['actions'] = [
        '#type' => 'actions',
      ];

      $form['existing_feedback']['actions']['edit_feedback'] = [
        '#type' => 'button',
        '#value' => $this->t('Edit Feedback'),
        '#ajax' => [
          'disable-refocus' => TRUE,
          'callback' => [$this, 'ajaxToggleFeedbackEdit'],
          'wrapper' => 'feedback-container-' . $rid,
        ],
        '#attributes' => [
          'class' => ['btn', 'btn-secondary'],
        ],
        '#name' => 'edit_feedback_' . $rid,
      ];

      $form['edit_form'] = [
        '#type' => 'container',
        '#attributes' => [
          'id' => 'edit-feedback-form-' . $rid,
          'style' => 'display: none;',
        ],
      ];

      $form['edit_form']['quality_rating_' . $rid] = [
        '#type' => 'radios',
        '#title' => $this->t('How would you rate the quality of the recommendations?'),
        '#default_value' => $report['feedback_quality_rating'],
        '#options' => $quality_rating_options,
      ];

      $form['edit_form']['feedback_comments_' . $rid] = [
        '#type' => 'textarea',
        '#title' => $this->t('What was not accurate or useful in the recommendations?'),
        '#rows' => 4,
        '#description' => $this->t('Please provide specific feedback about what was inaccurate or not useful.'),
        '#default_value' => $report['feedback_comments'] ?? '',
      ];

      $form['edit_form']['feedback_author_name_' . $rid] = [
        '#type' => 'textfield',
        '#title' => $this->t('Your name'),
        '#default_value' => $report['feedback_author_name'] ?? $this->currentUser->getDisplayName(),
        '#description' => $this->t('Name of person providing feedback.'),
      ];

      $form['edit_form']['report_id_' . $rid] = [
        '#type' => 'hidden',
        '#value' => $rid,
      ];

      $form['edit_form']['actions'] = [
        '#type' => 'actions',
      ];

      $form['edit_form']['actions']['update_feedback'] = [
        '#type' => 'submit',
        '#value' => $this->t('Update Feedback'),
        '#submit' => [[$this, 'submitFeedback']],
        '#validate' => [[$this, 'validateFeedback']],
        '#ajax' => [
          'disable-refocus' => TRUE,
          'callback' => [$this, 'ajaxToggleFeedbackEdit'],
          'wrapper' => 'feedback-container-' . $rid,
        ],
        '#attributes' => [
          'class' => ['btn', 'btn-primary'],
        ],
        '#name' => 'update_feedback_' . $rid,
      ];

      $form['edit_form']['actions']['cancel'] = [
        '#type' => 'button',
        '#value' => $this->t('Cancel'),
        '#ajax' => [
          'disable-refocus' => TRUE,
          'callback' => [$this, 'ajaxToggleFeedbackEdit'],
          'wrapper' => 'feedback-container-' . $rid,
        ],
        '#attributes' => [
          'class' => ['btn', 'btn-secondary'],
        ],
        '#name' => 'cancel_feedback_' . $rid,
      ];
    }
    else {
      // Show new feedback form
      $form['quality_rating_' . $rid] = [
        '#type' => 'radios',
        '#title' => $this->t('How would you rate the quality of the recommendations?'),
        '#options' => $quality_rating_options,
      ];

      $form['feedback_comments_' . $rid] = [
        '#type' => 'textarea',
        '#title' => $this->t('What was not accurate or useful in the recommendations?'),
        '#rows' => 4,
        '#description' => $this->t('Please provide specific feedback about what was inaccurate or not useful.'),
      ];

      $form['feedback_author_name_' . $rid] = [
        '#type' => 'textfield',
        '#title' => $this->t('Your name'),
        '#default_value' => $this->currentUser->getDisplayName(),
        '#description' => $this->t('Name of person providing feedback.'),
      ];

      $form['report_id_' . $rid] = [
        '#type' => 'hidden',
        '#value' => $rid,
      ];

      $form['actions'] = [
        '#type' => 'actions',
      ];

      $form['actions']['submit_feedback_' . $rid] = [
        '#name' => 'submit_feedback_' . $rid,
        '#type' => 'submit',
        '#value' => $this->t('Submit Feedback'),
        '#submit' => [[$this, 'submitFeedback']],
        '#validate' => [[$this, 'validateFeedback']],
        '#ajax' => [
          'disable-refocus' => TRUE,
          'callback' => [$this, 'ajaxToggleFeedbackEdit'],
          'wrapper' => 'feedback-container-' . $rid,
        ],
      ];
    }

    return $form;
  }

  /**
   * Builds the display for existing feedback.
   *
   * @param array $report
   *   The report data array.
   *
   * @return string
   *   HTML markup for existing feedback.
   */
  private function buildExistingFeedbackDisplay(array $report): string {
    $quality_options = [
      'very_low' => $this->t('Very low'),
      'low' => $this->t('Low'),
      'fair' => $this->t('Fair'),
      'good' => $this->t('Good'),
      'very_good' => $this->t('Very good'),
    ];

    $quality_rating = $quality_options[$report['feedback_quality_rating']] ?? $report['feedback_quality_rating'];
    $feedback_time = !empty($report['feedback_timestamp']) ? $this->dateFormatter->format($report['feedback_timestamp']) : '';

    $html = '<div class="existing-feedback">';
    $html .= '<h4>' . $this->t('Feedback Provided') . '</h4>';
    $html .= '<div class="feedback-rating"><strong>' . $this->t('Quality Rating:') . '</strong> ' . $quality_rating . '</div>';

    if (!empty($report['feedback_comments'])) {
      $html .= '<div class="feedback-comments"><strong>' . $this->t('Comments:') . '</strong><br>' . nl2br(htmlspecialchars($report['feedback_comments'])) . '</div>';
    }

    if (!empty($report['feedback_author_name'])) {
      $html .= '<div class="feedback-author"><strong>' . $this->t('Provided by:') . '</strong> ' . htmlspecialchars($report['feedback_author_name']) . '</div>';
    }

    if ($feedback_time) {
      $html .= '<div class="feedback-timestamp"><strong>' . $this->t('Submitted:') . '</strong> ' . $feedback_time . '</div>';
    }

    $html .= '</div>';

    return $html;
  }

  /**
   * Validates feedback form inputs and ensures required fields are populated.
   *
   * @param array $form
   *   The form structure as an associative array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form, including user-submitted values.
   *
   * @return void
   *   This method does not return a value but sets validation errors if any required fields are missing.
   */
  public function validateFeedback(array &$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    $button_name = $triggering_element['#name'] ?? '';
    $rid = NULL;
    // Extract report ID from button name (e.g., 'edit_feedback_123' or 'cancel_feedback_123')
    if (preg_match('/(?:update|submit)_feedback_(\d+)/', $button_name, $matches)) {
      $rid = $matches[1];
    }

    $quality_rating = $form_state->getValue('quality_rating_' . $rid);
    $feedback_comments = $form_state->getValue('feedback_comments_' . $rid);
    $feedback_author_name = $form_state->getValue('feedback_author_name_' . $rid);

    if (empty($quality_rating)) {
      $form_state->setErrorByName('quality_rating_' . $rid, $this->t('Please select a quality rating.'));
    }

    if (empty($feedback_comments)) {
      $form_state->setErrorByName('feedback_comments_' . $rid, $this->t('Please provide feedback.'));
    }

    if (empty($feedback_author_name)) {
      $form_state->setErrorByName('feedback_author_name_' . $rid, $this->t('Please provide your name.'));
    }

    if ($form_state->hasAnyErrors()) {
      $form_state->setRebuild();
    }

  }

  /**
   * Submit handler for feedback form.
   */
  public function submitFeedback(array &$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    $button_name = $triggering_element['#name'] ?? '';
    $rid = NULL;
    // Extract report ID from button name (e.g., 'edit_feedback_123' or 'cancel_feedback_123')
    if (preg_match('/(?:update|submit)_feedback_(\d+)/', $button_name, $matches)) {
      $rid = $matches[1];
    }

    $report_id = $form_state->getValue('report_id_' . $rid);
    $quality_rating = $form_state->getValue('quality_rating_' . $rid);
    $feedback_comments = $form_state->getValue('feedback_comments_' . $rid);
    $feedback_author_name = $form_state->getValue('feedback_author_name_' . $rid);

    if (!empty($quality_rating) || !empty($feedback_comments)) {
      $this->saveFeedback($report_id, $quality_rating, $feedback_comments, $feedback_author_name);
      $this->messenger()->addMessage($this->t('Thank you for your feedback!'));
      $form_state->setRebuild();
    }
  }

  /**
   * AJAX callback for toggling feedback edit mode.
   */
  public function ajaxToggleFeedbackEdit(array &$form, FormStateInterface $form_state) {
    // Get the triggering element to determine which report feedback we're dealing with
    $triggering_element = $form_state->getTriggeringElement();
    $button_name = $triggering_element['#name'] ?? '';

    // Extract report ID from button name (e.g., 'edit_feedback_123' or 'cancel_feedback_123')
    if (preg_match('/(?:edit|cancel|update|submit)_feedback_(\d+)/', $button_name, $matches)) {
      $rid = $matches[1];

      // Find the feedback container in the form structure
      $feedback_container = NULL;

      // Navigate through the form structure to find the feedback form.
      if (!empty($form['container']['reports']['feedback']['#attributes']['id']) &&
          $form['container']['reports']['feedback']['#attributes']['id'] === 'feedback-container-' . $rid) {
        $feedback_container = &$form['container']['reports']['feedback'];
      }

      if (!empty($form['container']['older_reports']['report_' . $rid]['feedback'])) {
        $feedback_container = &$form['container']['older_reports']['report_' . $rid]['feedback'];
      }

      if ($feedback_container !== NULL) {
        if (str_starts_with($button_name, 'edit_feedback')) {
          // Show edit form, hide display
          $feedback_container['existing_feedback']['#attributes']['style'] = 'display: none;';
          $feedback_container['edit_form']['#attributes']['style'] = 'display: block;';
        }
        elseif (str_starts_with($button_name, 'cancel_feedback')) {
          // Show display, hide an edit form
          $feedback_container['existing_feedback']['#attributes']['style'] = 'display: block;';
          $feedback_container['edit_form']['#attributes']['style'] = 'display: none;';
        }

        return $feedback_container;
      }
    }

    // Fallback - return the whole form if we can't find the specific container
    return $form;
  }

  /**
   * Saves feedback to the database.
   *
   * @param int $report_id
   *   The report ID.
   * @param string|null $quality_rating
   *   The quality rating.
   * @param string|null $feedback_comments
   *   The feedback comments.
   * @param string|null $feedback_author_name
   *   The feedback author name.
   */
  private function saveFeedback(int $report_id, string|null $quality_rating, string|null $feedback_comments, string|null $feedback_author_name): void {
    try {
      $this->entityTypeManager
        ->getStorage('ai_content_advisor_report')->load($report_id)
        ->set('feedback_quality_rating', $quality_rating)
        ->set('feedback_comments', $feedback_comments)
        ->set('feedback_author_name', $feedback_author_name)
        ->set('feedback_timestamp', \Drupal::time()->getRequestTime())
        ->save();
    }
    catch (\Exception $e) {
      $this->logger('ai_content_advisor')->error('Could not save feedback: @message', ['@message' => $e->getMessage()]);
      $this->messenger()->addError($this->t('An error occurred while saving your feedback.'));
    }
  }

  /**
   * Builds the HTML analyzed field with ace_editor support when available.
   *
   * @param string $value
   *   The field value.
   * @param string $report_type_id
   *   The report type ID.
   *
   * @return array
   *   The field render array.
   */
  private function buildHtmlAnalyzedField(string $value): array {
    $field = [
      '#type' => 'textarea',
      '#title' => 'HTML Analyzed',
      '#disabled' => TRUE,
      '#readonly' => TRUE,
      '#value' => $value,
      '#description' => $this->t('HTML analyzed for this report. Contains 2 versions of HTML: "Full HTML" with tags as it was received by from Drupal, and "Cleaned HTML" after internal processing needed before sending to the selected Chat.'),
      '#rows' => 10,
    ];

    // Check if ace_editor is installed and the content is valid HTML.
    if ($this->shouldUseAceEditor($value)) {
      $settings = [
        'theme' => 'cobalt',
        'syntax' => 'html',
        'height' => '600px',
        'width' => '100%',
        'font_size' => '12pt',
        'line_numbers' => '1',
        'print_margins' => '1',
        'use_wrap_mode' => '1',
        'show_invisibles' => 0,
        'fold_style' => 'manual',
      ];

      $field += [
        '#attached' => [
          'library' => [
            'ace_editor/formatter',
          ],
          'drupalSettings' => [
            // Pass settings variable ace_formatter to javascript.
            'ace_formatter' => $settings,
          ],
        ],
        '#attributes' => [
          'class' => ['content'],
          'readonly' => 'readonly',
        ],
        '#prefix' => '<div class="ace_formatter">',
        '#suffix' => '</div>',
      ];
    }

    return $field;
  }

  /**
   * detect if string is valid HTML.
   *
   * @param string $content
   *
   * @return bool
   */
  function isValidHtml(string $content): bool {
    if (empty($content) || !is_string($content)) {
      return false;
    }

    // Remove whitespace to avoid false positives
    $trimmed = trim($content);
    if (empty($trimmed)) {
      return false;
    }

    // Check for basic HTML indicators first
    if (!preg_match('/<[^>]+>/', $trimmed)) {
      return false;
    }

    // Use DOMDocument to parse and validate
    $dom = new \DOMDocument();

    // Suppress errors during parsing
    $previousErrorLevel = libxml_use_internal_errors(true);
    libxml_clear_errors();

    // Try to load the HTML
    $result = $dom->loadHTML('<!DOCTYPE html><html><body>' . $trimmed . '</body></html>',
      LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);

    // Get any parsing errors
    $errors = libxml_get_errors();

    // Restore previous error level
    libxml_use_internal_errors($previousErrorLevel);
    libxml_clear_errors();

    // Consider it HTML if it loaded successfully (even with errors)
    // and doesn't have fatal errors
    if ($result) {
      // Check if there are any fatal errors
      foreach ($errors as $error) {
        if ($error->level === LIBXML_ERR_FATAL) {
          return false;
        }
      }
      return true;
    }

    return false;
  }

  /**
   * Determines if ace_editor should be used based on field content.
   *
   * @param string $content
   *
   * @return bool
   *   TRUE if ace_editor should be used, FALSE otherwise.
   */
  private function shouldUseAceEditor(string $content): bool {
    // Check if ace_editor module is installed.
    if (!$this->moduleHandler->moduleExists('ace_editor')) {
      return FALSE;
    }

    try {
      return $this->isValidHtml($content);
    }
    catch (\Exception $e) {
      $this->logger('ai_content_advisor')->error('Could not determine if ace_editor should be used: @message', ['@message' => $e->getMessage()]);;
    }

    return FALSE;
  }

}
