#!/bin/bash

set -vo pipefail

# Install required libs for Drupal
GD_ENABLED=$(php -i | grep 'GD Support' | awk '{ print $4 }')

if [ "$GD_ENABLED" != 'enabled' ]; then
  apk update && \
  apk add libpng libpng-dev libjpeg-turbo-dev libwebp-dev zlib-dev libxpm-dev gd tree rsync && docker-php-ext-install gd
fi

# Create project in a temporary directory inside the container
INSTALL_DIR="/drupal_install_tmp"
composer create-project drupal/recommended-project:11.x-dev "$INSTALL_DIR" --no-interaction --stability=dev

cd "$INSTALL_DIR"

# Allow specific plugins needed by dependencies before requiring them.
composer config --no-plugins allow-plugins.tbachert/spi true --no-interaction

# Create phpstan.neon config file
cat <<EOF > phpstan.neon
parameters:
    paths:
        - web/modules/contrib/ai_content_strategy
    # Set the analysis level (0-9)
    level: 5
    # Don't treat PHPDoc types as certain to reduce false positives
    treatPhpDocTypesAsCertain: false
    ignoreErrors:
        # AI module ProviderProxy methods - dynamic magic methods
        - '#Call to an undefined method Drupal\\\\ai\\\\Plugin\\\\ProviderProxy::(chat|isUsable|getConfiguredModels|setChatSystemRole)\(\)#'
        # new static() is intentional for factory pattern
        - '#Unsafe usage of new static\(\)#'
EOF

mkdir -p web/modules/contrib/

if [ ! -L "web/modules/contrib/ai_content_strategy" ]; then
  ln -s /src web/modules/contrib/ai_content_strategy
fi

# Install the statistics module if D11 (removed from core).
composer require drupal/statistics --no-interaction

# Install required dependencies for ai_content_strategy module
composer require drupal/ai --no-interaction

# Install PHPStan extensions for Drupal 11 and Drush for command analysis
composer require --dev phpstan/phpstan mglaman/phpstan-drupal phpstan/phpstan-deprecation-rules drush/drush --with-all-dependencies --no-interaction

# Run phpstan
./vendor/bin/phpstan analyse --memory-limit=-1 -c phpstan.neon
