<?php

declare(strict_types=1);

namespace Drupal\ai_context\Form;

use Drupal\ai\AiProviderPluginManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure AI Context settings.
 */
final class AiContextSettingsForm extends ConfigFormBase {

  /**
   * Constructs an AiContextSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\ai\AiProviderPluginManager $aiProvider
   *   The AI provider plugin manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typedConfigManager,
    protected AiProviderPluginManager $aiProvider,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('ai.provider'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ai_context_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['ai_context.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('ai_context.settings');

    $form['strategy'] = [
      '#type' => 'select',
      '#title' => $this->t('Selection strategy'),
      '#options' => [
        'keyword' => $this->t('Keyword (default)'),
        'llm' => $this->t('LLM-assisted (requires provider/model)'),
      ],
      '#default_value' => $config->get('strategy') ?? 'keyword',
    ];

    $form['max_contexts'] = [
      '#type' => 'number',
      '#title' => $this->t('Max contexts to inject'),
      '#default_value' => $config->get('max_contexts') ?? 3,
      '#min' => 1,
      '#max' => 10,
    ];

    $form['max_tokens'] = [
      '#type' => 'number',
      '#title' => $this->t('Token budget for injection'),
      '#default_value' => $config->get('max_tokens') ?? 1200,
      '#min' => 100,
      '#description' => $this->t('Estimated tokens reserved for context block appended to sub-agent system prompts.'),
    ];

    // Build provider/model options for the 'chat' operation.
    // This is broadly supported and sufficient for LLM-assisted routing.
    $options = $this->aiProvider->getSimpleProviderModelOptions('chat', TRUE, TRUE, []);
    $selected_combo = '';
    $saved_provider = (string) ($config->get('provider_id') ?? '');
    $saved_model = (string) ($config->get('model_id') ?? '');
    if ($saved_provider && $saved_model) {
      $candidate = $saved_provider . '__' . $saved_model;
      if (isset($options[$candidate])) {
        $selected_combo = $candidate;
      }
    }
    $form['provider_model'] = [
      '#type' => 'select',
      '#title' => $this->t('Preferred Provider/Model (LLM)'),
      '#options' => $options,
      '#default_value' => $selected_combo,
      '#states' => [
        'visible' => [
          ':input[name="strategy"]' => ['value' => 'llm'],
        ],
      ],
      '#description' => $this->t('Lists configured providers and their models for the chat_with_tools operation.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $strategy = (string) $form_state->getValue('strategy');
    $provider_id = '';
    $model_id = '';
    if ($strategy === 'llm') {
      $combo = (string) $form_state->getValue('provider_model');
      if ($combo) {
        $parts = explode('__', $combo);
        if (count($parts) === 2) {
          $provider_id = $parts[0];
          $model_id = $parts[1];
        }
      }
    }

    $this->configFactory->getEditable('ai_context.settings')
      ->set('strategy', $strategy)
      ->set('max_contexts', (int) $form_state->getValue('max_contexts'))
      ->set('max_tokens', (int) $form_state->getValue('max_tokens'))
      ->set('provider_id', $provider_id)
      ->set('model_id', $model_id)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
