<?php

declare(strict_types=1);

namespace Drupal\ai_context\EventSubscriber;

use Drupal\ai_agents\Event\BuildSystemPromptEvent;
use Drupal\ai_context\Service\AiContextSelector;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber to append relevant context to AI agent system prompts.
 */
final class SystemPromptSubscriber implements EventSubscriberInterface {

  /**
   * Constructs a SystemPromptSubscriber object.
   *
   * @param \Drupal\ai_context\Service\AiContextSelector $selector
   *   The AI context selector service.
   */
  public function __construct(private readonly AiContextSelector $selector) {}

  /**
   * {@inheritdoc}
   *
   * @return array
   *   The subscribed events.
   */
  public static function getSubscribedEvents(): array {
    return [
      BuildSystemPromptEvent::EVENT_NAME => 'onPreSystemPrompt',
    ];
  }

  /**
   * Responds to the pre-system prompt event.
   *
   * Selects and appends relevant AI context to the system prompt based on
   * the agent ID and existing prompt content.
   *
   * @param \Drupal\ai_agents\Event\BuildSystemPromptEvent $event
   *   The build system prompt event.
   */
  public function onPreSystemPrompt(BuildSystemPromptEvent $event): void {
    $agentId = $event->getAgentId();
    $prompt = $event->getSystemPrompt();
    if ($agentId && $prompt) {
      $result = $this->selector->select($prompt, $agentId);
      if (!empty($result['text'])) {
        $contextPrompt = (string) $result['text'];
      }
    }
    if (!isset($contextPrompt)) {
      return;
    }
    $append = "\n\nThe following site-specific context applies to this task. Use it strictly when relevant; do not override user intent.\n";
    $append .= "-----------------------------------------------\n";
    $append .= $contextPrompt;
    $append .= "-----------------------------------------------\n";
    $event->setSystemPrompt($prompt . $append);
  }

}
