<?php

declare(strict_types=1);

namespace Drupal\ai_document_ocr\OperationType\DocumentToText;

use Drupal\ai\OperationType\InputBase;

/**
 * Input for Document to Text operations (OCR).
 */
class DocumentToTextInput extends InputBase {

  /**
   * The document content (base64 encoded).
   *
   * @var string
   */
  protected string $documentContent;

  /**
   * The MIME type of the document.
   *
   * @var string
   */
  protected string $mimeType;

  /**
   * The filename of the document.
   *
   * @var string|null
   */
  protected ?string $filename;

  /**
   * Constructor.
   *
   * @param string $document_content
   *   The document content (base64 encoded).
   * @param string $mime_type
   *   The MIME type of the document.
   * @param string|null $filename
   *   The filename of the document.
   */
  public function __construct(string $document_content, string $mime_type, ?string $filename = NULL) {
    $this->documentContent = $document_content;
    $this->mimeType = $mime_type;
    $this->filename = $filename;
  }

  /**
   * Gets the document content.
   *
   * @return string
   *   The base64 encoded document content.
   */
  public function getDocumentContent(): string {
    return $this->documentContent;
  }

  /**
   * Gets the MIME type.
   *
   * @return string
   *   The MIME type.
   */
  public function getMimeType(): string {
    return $this->mimeType;
  }

  /**
   * Gets the filename.
   *
   * @return string|null
   *   The filename or NULL.
   */
  public function getFilename(): ?string {
    return $this->filename;
  }

  /**
   * Convert object to string representation.
   */
  public function toString(): string {
    return sprintf(
      'DocumentToTextInput[%s, %s, %d bytes]',
      $this->mimeType,
      $this->filename ?? 'unknown',
      strlen($this->documentContent)
    );
  }

}
