<?php

declare(strict_types=1);

namespace Drupal\ai_document_ocr\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\key\KeyRepositoryInterface;
use Drupal\ai\Service\AiProviderFormHelper;
use Firebase\JWT\JWT;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure AI Document OCR Provider settings.
 */
class AiProviderConfigForm extends ConfigFormBase {

  /**
   * Config settings.
   */
  const CONFIG_NAME = 'ai_document_ocr.settings';

  /**
   * Constructs a new AI Document OCR Provider Config object.
   */
  final public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    protected AiProviderFormHelper $formHelper,
    protected ?KeyRepositoryInterface $keyRepository,
    protected ModuleHandlerInterface $moduleHandler,
    protected ClientInterface $httpClient,
    LoggerChannelFactoryInterface $loggerFactory,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->loggerFactory = $loggerFactory;
  }

  /**
   * {@inheritdoc}
   */
  final public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('ai.form_helper'),
      $container->has('key.repository') ? $container->get('key.repository') : NULL,
      $container->get('module_handler'),
      $container->get('http_client'),
      $container->get('logger.factory'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ai_document_ocr_provider_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [static::CONFIG_NAME];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config(static::CONFIG_NAME);
    // Add AI Provider selection.
    $this->formHelper->generateAiProvidersForm(
      $form,
      $form_state,
      'document_to_text',
      'ai_document_ocr',
      AiProviderFormHelper::FORM_CONFIGURATION_NONE,
      0,
      '',
      $this->t('AI Provider'),
      $this->t('Select the AI provider to use for document OCR operations.')
    );

    // Provider configuration section.
    $form['provider_config'] = [
      '#type' => 'details',
      '#title' => $this->t('Google Document AI Configuration'),
      '#description' => $this->t('Configure your Google Cloud Document AI settings. <a href="@url" target="_blank">Learn how to set up Google Cloud Document AI</a>.', ['@url' => 'https://cloud.google.com/document-ai/docs/setup']),
      '#open' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="ai_provider"]' => ['value' => 'ai_document_ocr'],
        ],
      ],
    ];

    // Credentials.
    if ($this->keyRepository) {
      $form['provider_config']['general_credentials_file'] = [
        '#type' => 'key_select',
        '#title' => $this->t('Google Cloud Service Account Key'),
        '#key_filters' => ['provider' => 'file'],
        '#description' => $this->t('Select the Google Cloud service account JSON key file. <strong>Make sure it has "Document AI API User" role</strong> and access to your processors.'),
        '#default_value' => $config->get('general_credentials_file'),
        '#required' => TRUE,
        '#ajax' => [
          'callback' => '::loadProcessorsCallback',
          'wrapper' => 'processors-wrapper',
          'effect' => 'fade',
          'event' => 'change',
        ],
      ];
    }
    else {
      $form['provider_config']['no_key_module'] = [
        '#markup' => '<div class="messages messages--error">' .
        $this->t('The <a href="@url" target="_blank">Key module</a> is required for secure credential storage.', ['@url' => 'https://www.drupal.org/project/key']) .
        '</div>',
      ];
    }

    // Info message about automatic region detection.
    $form['provider_config']['region_info'] = [
      '#markup' => '<div class="description" style="background: #f0f8ff; padding: 10px; border-left: 4px solid #0073aa; margin: 10px 0;">' .
      $this->t('<strong>Automatic Region Detection:</strong> Processors from all Google Cloud regions will be loaded automatically. The system will detect the correct region for each processor.') .
      '</div>',
    ];

    // Processors container for AJAX updates.
    $form['provider_config']['processors'] = [
      '#type' => 'container',
      '#prefix' => '<div id="processors-wrapper">',
      '#suffix' => '</div>',
    ];

    // Check if we have credentials from form state or saved config.
    $key_id = $form_state->getValue('general_credentials_file') ?: $config->get('general_credentials_file');
    $processor_id = $config->get('processor_id');

    if (!empty($key_id)) {
      // If we have both key and processor saved,
      // trigger AJAX to reload processors.
      $project_id = $this->getProjectIdFromKey($key_id);
      if (!empty($project_id)) {
        $processors = $this->loadAvailableProcessors($project_id, '', $key_id);

        if (!empty($processors)) {
          $form['provider_config']['processors']['processor_id'] = [
            '#type' => 'select',
            '#title' => $this->t('Select Document AI Processor'),
            '#description' => $this->t('Choose the Document AI processor to use for OCR operations. Processors are shown with their region for clarity.'),
            '#options' => $processors,
            '#default_value' => $processor_id ?? '',
            '#empty_option' => $this->t('- Select a processor -'),
            '#required' => TRUE,
          ];
        }
        else {
          $form['provider_config']['processors']['no_processors'] = [
            '#markup' => '<div class="messages messages--warning">' .
            $this->t('No enabled processors found in any region. <a href="@url" target="_blank">Create processors in Google Cloud Console</a>.', ['@url' => 'https://console.cloud.google.com/ai/document-ai']) .
            '</div>',
          ];
        }
      }
      else {
        $form['provider_config']['processors']['error'] = [
          '#markup' => '<div class="messages messages--error">' .
          $this->t('Invalid credentials key. Please select a valid Google Cloud service account key.') .
          '</div>',
        ];
      }
    }
    else {
      // Show instructions for first-time setup.
      $form['provider_config']['processors']['instructions'] = [
        '#markup' => '<div class="description">' .
        $this->t('Select your Google Cloud service account key above to automatically load all available processors.') .
        '</div>',
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Skip validation for AJAX requests.
    if ($form_state->getTriggeringElement() &&
        isset($form_state->getTriggeringElement()['#ajax'])) {
      return;
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Get processor_id from nested form structure.
    $processor_id = $form_state->getValue(['processors', 'processor_id']) ?: $form_state->getValue('processor_id');

    $this->config(static::CONFIG_NAME)
      ->set('general_credentials_file', $form_state->getValue('general_credentials_file'))
      ->set('processor_id', $processor_id)
      ->save();

    // Add success message.
    if ($processor_id) {
      $this->messenger()->addStatus($this->t('Document OCR provider configured successfully with processor: @processor', ['@processor' => $processor_id]));
    }

    parent::submitForm($form, $form_state);
  }

  /**
   * AJAX callback for loading processors when credentials change.
   */
  public function loadProcessorsCallback(array &$form, FormStateInterface $form_state): array {
    $key_id = $form_state->getValue('general_credentials_file');
    $project_id = $this->getProjectIdFromKey($key_id);

    $form['provider_config']['processors'] = [
      '#type' => 'container',
      '#prefix' => '<div id="processors-wrapper">',
      '#suffix' => '</div>',
    ];

    if (empty($project_id) || empty($key_id)) {
      $form['provider_config']['processors']['instructions'] = [
        '#markup' => '<div class="description">' .
        $this->t('Select your Google Cloud service account key to automatically load all available processors.') .
        '</div>',
      ];
      return $form['provider_config']['processors'];
    }

    // Show loading message first.
    $form['provider_config']['processors']['loading'] = [
      '#markup' => '<div class="description">' .
      $this->t('Loading processors from all regions... This may take a moment.') .
      '</div>',
    ];

    $processors = $this->loadAvailableProcessors($project_id, '', $key_id);

    // Remove loading message.
    unset($form['provider_config']['processors']['loading']);

    if (!empty($processors)) {
      $form['provider_config']['processors']['processor_id'] = [
        '#type' => 'select',
        '#title' => $this->t('Select Document AI Processor'),
        '#description' => $this->t('Choose the Document AI processor to use for OCR operations. Processors are shown with their region for clarity.'),
        '#options' => $processors,
        '#empty_option' => $this->t('- Select a processor -'),
        '#required' => TRUE,
      ];

      $form['provider_config']['processors']['success'] = [
        '#markup' => '<div class="messages messages--status">' .
        $this->t('Found @count processor(s) across all regions.', ['@count' => count($processors)]) .
        '</div>',
      ];
    }
    else {
      $form['provider_config']['processors']['no_processors'] = [
        '#markup' => '<div class="messages messages--warning">' .
        $this->t('No enabled processors found in any region. <a href="@url" target="_blank">Create processors in Google Cloud Console</a>.', ['@url' => 'https://console.cloud.google.com/ai/document-ai']) .
        '</div>',
      ];
    }

    return $form['provider_config']['processors'];
  }

  /**
   * Get available Google Cloud regions.
   */
  protected function getRegionOptions(): array {
    return [
      'us' => $this->t('United States'),
      'eu' => $this->t('Europe'),
      'us-central1' => $this->t('Iowa'),
      'us-east1' => $this->t('South Carolina'),
      'us-west1' => $this->t('Oregon'),
      'europe-west1' => $this->t('Belgium'),
      'europe-west2' => $this->t('London'),
      'europe-west4' => $this->t('Netherlands'),
      'asia-east1' => $this->t('Taiwan'),
      'asia-northeast1' => $this->t('Tokyo'),
      'asia-southeast1' => $this->t('Singapore'),
    ];
  }

  /**
   * Load available Document AI processors from all regions.
   */
  protected function loadAvailableProcessors(string $project_id, string $region = '', string $key_id = ''): array {
    $processors = [];

    if (empty($project_id) || empty($key_id)) {
      return $processors;
    }

    // Get regions from the existing region options.
    $regions = array_keys($this->getRegionOptions());

    // If a specific region is provided, prioritize it first.
    if (!empty($region) && !in_array($region, $regions)) {
      array_unshift($regions, $region);
    }

    try {
      $credentials = $this->loadCredentials($key_id);
      if (!$credentials) {
        return $processors;
      }

      $access_token = $this->getAccessToken($credentials);
      if (!$access_token) {
        return $processors;
      }

      // Query each region for processors.
      foreach ($regions as $current_region) {
        try {
          $endpoint_base = $this->getDocumentAiEndpoint($current_region);
          $endpoint = "{$endpoint_base}/v1/projects/{$project_id}/locations/{$current_region}/processors";

          $response = $this->httpClient->request('GET', $endpoint, [
            'headers' => [
              'Authorization' => 'Bearer ' . $access_token,
              'Content-Type' => 'application/json',
            ],
            // Shorter timeout for multiple requests.
            'timeout' => 10,
          ]);

          if ($response->getStatusCode() === 200) {
            $data = json_decode($response->getBody()->getContents(), TRUE);
            if (isset($data['processors']) && is_array($data['processors'])) {
              foreach ($data['processors'] as $processor) {
                if ($processor['state'] === 'ENABLED') {
                  $processor_id = basename($processor['name']);
                  $display_name = $processor['displayName'] ?? $processor_id;

                  // Include region in the display name for clarity.
                  $processors[$processor_id] = "{$display_name} ({$current_region})";
                }
              }
            }
          }
        }
        catch (\Exception $e) {
          // Continue to next region on failure.
          continue;
        }
      }
    }
    catch (\Exception $e) {
      // Return empty array on failure.
    }

    return $processors;
  }

  /**
   * Load credentials from Key module.
   */
  protected function loadCredentials(string $key_id): ?array {
    if (!$this->moduleHandler->moduleExists('key') || !$this->keyRepository) {
      return NULL;
    }

    try {
      $key = $this->keyRepository->getKey($key_id);
      if (!$key || !($file_content = $key->getKeyValue())) {
        return NULL;
      }

      $decoded = json_decode($file_content, TRUE);
      return $decoded ?: NULL;
    }
    catch (\Exception $e) {
      return NULL;
    }
  }

  /**
   * Get access token for Google Cloud API.
   */
  protected function getAccessToken(array $credentials): ?string {
    try {
      $payload = [
        'iss' => $credentials['client_email'],
        'scope' => 'https://www.googleapis.com/auth/cloud-platform',
        'aud' => 'https://oauth2.googleapis.com/token',
        'exp' => time() + 3600,
        'iat' => time(),
      ];

      $jwt = JWT::encode($payload, $credentials['private_key'], 'RS256');

      $response = $this->httpClient->request('POST', 'https://oauth2.googleapis.com/token', [
        'form_params' => [
          'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
          'assertion' => $jwt,
        ],
      ]);

      if ($response->getStatusCode() === 200) {
        $data = json_decode($response->getBody()->getContents(), TRUE);
        return $data['access_token'] ?? NULL;
      }
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('ai_document_ocr')->error(
        'Failed to get token:: @error',
        [
          '@error' => $e->getMessage(),
        ]
      );
      return NULL;
    }

    return NULL;
  }

  /**
   * Get Document AI endpoint for region.
   */
  protected function getDocumentAiEndpoint(string $location): string {
    return "https://{$location}-documentai.googleapis.com";
  }

  /**
   * Get project ID from Google Cloud credentials.
   *
   * @param string $key_id
   *   The key ID to load credentials from.
   *
   * @return string|null
   *   The project ID from the credentials file, or null if not found.
   */
  public function getProjectIdFromKey(string $key_id): ?string {
    if (empty($key_id) || !$this->keyRepository) {
      return NULL;
    }

    try {
      $key = $this->keyRepository->getKey($key_id);
      if (!$key || !($file_content = $key->getKeyValue())) {
        return NULL;
      }

      $credentials = json_decode($file_content, TRUE);
      return $credentials['project_id'] ?? NULL;
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('ai_document_ocr')->error(
        'Failed to get project_id: @error',
        [
          '@error' => $e->getMessage(),
        ]
      );
      return NULL;
    }
  }

}
