<?php

declare(strict_types=1);

namespace Drupal\ai_dropsolid\Service;

/**
 * Defines the contract for the recursive text chunker service.
 *
 * This interface extends the functionality of the core AI module's
 * TextChunkerInterface with Dropsolid-specific enhancements for
 * intelligent text processing.
 */
interface RecursiveTextChunkerInterface {

  /**
   * Chunks text into smaller pieces using recursive splitting strategies.
   *
   * This method converts token-based size constraints to character-based
   * limits and uses a hierarchical separator approach to maintain semantic
   * coherence while respecting size boundaries.
   *
   * @param string $text
   *   The input text to be chunked.
   * @param int $maxSize
   *   The maximum desired chunk size in tokens.
   * @param int $minOverlap
   *   The minimum desired overlap between chunks in tokens.
   *
   * @return array
   *   An array of text chunks, each respecting the configured size limits.
   *
   * @throws \InvalidArgumentException
   *   When parameters are invalid (negative values, overlap >= max size).
   */
  public function chunkText(string $text, int $maxSize, int $minOverlap): array;

  /**
   * Counts the number of tokens in the given text.
   *
   * @param string $text
   *   The text to count tokens for.
   *
   * @return int
   *   The number of tokens in the text.
   */
  public function countTokens(string $text): int;

  /**
   * Sets the tokenization model to use.
   *
   * @param string $model
   *   The model identifier to use for tokenization.
   */
  public function setModel(string $model): void;

}
