<?php

declare(strict_types=1);

namespace Drupal\ai_dropsolid\EventSubscriber;

use Drupal\ai\Event\PreGenerateResponseEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Injects LiteLLM metadata tags for AI operations with DXP-prefixed tagging.
 *
 * The subscriber reviews every AI event and applies a single DX-themed tag
 * that reflects the context of the operation. Using a priority of zero keeps
 * the behaviour predictable compared to other subscribers.
 */
final class LiteLlmTagSubscriber implements EventSubscriberInterface {

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      PreGenerateResponseEvent::EVENT_NAME => ['onPreGenerateResponse', 0],
    ];
  }

  /**
   * Adds LiteLLM metadata tags to all AI operations with DXP-prefixed tagging.
   *
   * @param \Drupal\ai\Event\PreGenerateResponseEvent $event
   *   The pre-generate event.
   */
  public function onPreGenerateResponse(PreGenerateResponseEvent $event): void {
    if ($event->getProviderId() !== 'litellm') {
      return;
    }

    $configuration = $event->getConfiguration();
    $metadata = $configuration['metadata'] ?? [];

    // Generate a single DXP-prefixed tag based on the operation context.
    $dxpTag = $this->generateDxpTag($event);

    if ($dxpTag !== NULL) {
      $metadata['tags'] = [$dxpTag];
    }
    else {
      // No applicable tag - remove tags from metadata.
      unset($metadata['tags']);
    }

    // Update configuration with metadata or remove empty metadata.
    if (!empty($metadata)) {
      $configuration['metadata'] = $metadata;
    }
    else {
      unset($configuration['metadata']);
    }

    $event->setConfiguration($configuration);
  }

  /**
   * Generates a single DXP-prefixed tag based on AI operation context.
   *
   * This method analyzes the event context (operation type, existing tags,
   * and so on) and returns a standardized tag following the format
   * `dxp_{feature}`.
   *
   * Supported features:
   * - chat: For conversational AI operations
   * - embeddings: For text embedding operations
   * - completion: For text completion operations
   * - search: For search-related AI operations
   * - analysis: For content analysis operations
   *
   * @param \Drupal\ai\Event\PreGenerateResponseEvent $event
   *   The AI operation event.
   *
   * @return string|null
   *   The DXP-prefixed tag or NULL if no applicable tag can be determined.
   */
  private function generateDxpTag(PreGenerateResponseEvent $event): ?string {
    $operationType = $event->getOperationType();
    $existingTags = $event->getTags();

    // If there's a dxp_ tag already, respect it (return the LAST dxp_ tag).
    foreach (array_reverse($existingTags) as $tag) {
      if (str_starts_with(strtolower(trim($tag)), 'dxp_')) {
        return strtolower(trim($tag));
      }
    }

    // If not, check the tag mappings.
    $tagMappings = [
      'ai_api_explorer' => 'dxp_ai_api_explorer',
      'ai_search' => 'dxp_ai_search',
      'ai_search_block' => 'dxp_ai_search',
      'ai_assistant_api' => 'dxp_ai_assistant',
      'ai_agents' => 'dxp_ai_agents',
      'ai_translate' => 'dxp_ai_translation',
      'ai_ckeditor' => 'dxp_ai_content_creation',
    ];

    // Look for feature-specific tags using array intersection for brevity.
    $normalizedTags = array_map(
      static fn ($tag) => strtolower(trim($tag)),
      $existingTags,
    );
    $matchingKeys = array_intersect($normalizedTags, array_keys($tagMappings));
    if (!empty($matchingKeys)) {
      return $tagMappings[reset($matchingKeys)];
    }

    // Fall back to operation type specific tagging when available.
    $operationMappings = [
      'chat' => 'dxp_ai_chat',
      'embeddings' => 'dxp_ai_embeddings',
      'completion' => 'dxp_ai_completion',
      'search' => 'dxp_ai_search',
      'analysis' => 'dxp_ai_analysis',
    ];

    if (isset($operationMappings[$operationType])) {
      return $operationMappings[$operationType];
    }

    // No applicable tag found.
    return NULL;
  }

}
