<?php

declare(strict_types=1);

namespace Drupal\Tests\ai_dropsolid\Unit\Tokenizer;

use Drupal\ai\Utility\TokenizerInterface;
use Drupal\ai_dropsolid\Tokenizer\DropsolidTokenizerDecorator;
use Drupal\ai_dropsolid\Tokenizer\DropsolidXlmRobertaTokenizer;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\key\KeyRepositoryInterface;
use GuzzleHttp\ClientInterface;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;

/**
 * @coversDefaultClass \Drupal\ai_dropsolid\Tokenizer\DropsolidTokenizerDecorator
 */
final class DropsolidTokenizerDecoratorTest extends TestCase {

  /**
   * @covers ::setModel
   * @covers ::countTokens
   * @covers ::getTokens
   * @covers ::getSupportedModels
   */
  public function testDelegatesWhenModelIsNotDropsolid(): void {
    $inner = new TestTokenizer(['inner__chat' => 'Inner Chat Model']);
    $decorator = new DropsolidTokenizerDecorator($inner, $this->createDropsolidTokenizer([
      'tokenizer.mode' => 'none',
    ]));

    $decorator->setModel('gpt-3.5');

    self::assertSame('gpt-3.5', $inner->activeModel);
    self::assertSame(104, $decorator->countTokens('test'));
    self::assertSame(['test'], $decorator->getTokens('test'));

    $supported = $decorator->getSupportedModels();
    self::assertArrayHasKey('inner__chat', $supported);
  }

  /**
   * @covers ::setModel
   * @covers ::countTokens
   * @covers ::getEncodedChunks
   * @covers ::decodeChunk
   */
  public function testSwitchesToDropsolidTokenizerWhenModelSupported(): void {
    $inner = new TestTokenizer(['inner__chat' => 'Inner Chat Model']);
    $dropsolidTokenizer = $this->createDropsolidTokenizer([
      'tokenizer.mode' => 'none',
    ]);
    $decorator = new DropsolidTokenizerDecorator($inner, $dropsolidTokenizer);

    // Prime the decorator so the inner tokenizer is used first.
    $decorator->setModel('gpt-3.5');

    $decorator->setModel('xlm-roberta-base');

    self::assertSame('xlm-roberta-base', $dropsolidTokenizer->getActiveModel());
    self::assertSame(4, $decorator->countTokens('robo'));

    $chunks = $decorator->getEncodedChunks('roberta', 3);
    self::assertCount(3, $chunks);

    $reconstructed = '';
    foreach ($chunks as $chunk) {
      $reconstructed .= $decorator->decodeChunk($chunk);
    }

    self::assertSame('roberta', $reconstructed);

    // Ensure the inner tokenizer remains on its previous model and was not
    // overridden by the Dropsolid pathway.
    self::assertSame('gpt-3.5', $inner->activeModel);
  }

  /**
   * Builds a Dropsolid tokenizer instance with stubbed dependencies.
   */
  private function createDropsolidTokenizer(array $configValues): DropsolidXlmRobertaTokenizer {
    $dropsolidConfig = $this->createMock(Config::class);
    $dropsolidConfig->method('get')->willReturnCallback(static function (string $key) use ($configValues) {
      return $configValues[$key] ?? NULL;
    });

    $liteConfig = $this->createMock(Config::class);
    $liteConfig->method('get')->willReturn(NULL);

    $configFactory = $this->createMock(ConfigFactoryInterface::class);
    $configFactory
      ->method('get')
      ->willReturnMap([
        ['ai_dropsolid.settings', $dropsolidConfig],
        ['ai_provider_litellm.settings', $liteConfig],
      ]);

    $fileSystem = $this->createMock(FileSystemInterface::class);
    $fileSystem
      ->method('realpath')
      ->willReturnArgument(0);

    $storage = $this->createMock(EntityStorageInterface::class);
    $storage->method('load')->willReturn(NULL);

    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager
      ->method('getStorage')
      ->with('file')
      ->willReturn($storage);

    $keyRepository = $this->createMock(KeyRepositoryInterface::class);
    $keyRepository->method('getKey')->willReturn(NULL);

    $httpClient = $this->createMock(ClientInterface::class);

    $logger = $this->createMock(LoggerInterface::class);

    return new DropsolidXlmRobertaTokenizer($configFactory, $fileSystem, $entityTypeManager, $keyRepository, $httpClient, $logger);
  }

}

/**
 * Simple tokenizer implementation used for decorator tests.
 */
final class TestTokenizer implements TokenizerInterface {

  /**
   * The currently active model name.
   *
   * @var string
   */
  public string $activeModel = '';

  /**
   * Constructs a new TestTokenizer.
   *
   * @param array $supportedModels
   *   Models exposed through the select element.
   */
  public function __construct(private readonly array $supportedModels) {
  }

  /**
   * {@inheritdoc}
   */
  public function setModel(string $model): void {
    $this->activeModel = $model;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedModels(): array {
    return $this->supportedModels;
  }

  /**
   * {@inheritdoc}
   */
  public function getTokens(string $chunk): array {
    return [$chunk];
  }

  /**
   * {@inheritdoc}
   */
  public function countTokens(string $chunk): int {
    return 100 + strlen($chunk);
  }

  /**
   * Provides encoded chunks for the supplied text.
   */
  public function getEncodedChunks(string $text, int $maxSize): array {
    $tokens = str_split($text);
    return array_chunk($tokens, max(1, $maxSize));
  }

  /**
   * Decodes a chunk back into a string.
   */
  public function decodeChunk(array $encodedChunk): string {
    return implode('', $encodedChunk);
  }

}
