<?php

declare(strict_types=1);

namespace Drupal\Tests\ai_dropsolid\Unit\Tokenizer;

use Drupal\ai_dropsolid\Tokenizer\DropsolidXlmRobertaTokenizer;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\key\KeyRepositoryInterface;
use GuzzleHttp\ClientInterface;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;

/**
 * @coversDefaultClass \Drupal\ai_dropsolid\Tokenizer\DropsolidXlmRobertaTokenizer
 */
final class DropsolidXlmRobertaTokenizerTest extends TestCase {

  /**
   * @covers ::setModel
   * @covers ::getActiveModel
   * @covers ::supportsModel
   */
  public function testModelSelectionFallsBackToDefault(): void {
    $tokenizer = $this->createTokenizer([
      'tokenizer.mode' => 'none',
    ]);

    $tokenizer->setModel('unsupported-model');
    self::assertSame('xlm-roberta-base', $tokenizer->getActiveModel());

    $tokenizer->setModel('xlm-roberta-base');
    self::assertSame('xlm-roberta-base', $tokenizer->getActiveModel());
  }

  /**
   * @covers ::getSupportedModels
   */
  public function testSupportedModelsExposeChatOption(): void {
    $tokenizer = $this->createTokenizer([
      'tokenizer.mode' => 'none',
    ]);

    $supported = $tokenizer->getSupportedModels();

    self::assertArrayHasKey('dropsolid_xlmr__xlm-roberta-base', $supported);
  }

  /**
   * @covers ::getTokens
   * @covers ::countTokens
   * @covers ::getEncodedChunks
   * @covers ::decodeChunk
   */
  public function testTokenizationPreservesContent(): void {
    $tokenizer = $this->createTokenizer([
      'tokenizer.mode' => 'none',
    ]);
    $tokenizer->setModel('xlm-roberta-base');

    $text = 'XLM Roberta tokenizes multilingual text.';

    $tokens = $tokenizer->getTokens($text);
    self::assertSame(strlen($text), $tokenizer->countTokens($text));

    $chunks = $tokenizer->getEncodedChunks($text, 10);
    self::assertNotEmpty($chunks);

    $reconstructed = '';
    foreach ($chunks as $chunk) {
      $reconstructed .= $tokenizer->decodeChunk($chunk);
    }

    self::assertSame($text, $reconstructed);
    self::assertSame($tokens, array_merge(...$chunks));
  }

  /**
   * Creates a tokenizer instance with stubbed dependencies.
   */
  private function createTokenizer(array $configValues): DropsolidXlmRobertaTokenizer {
    $dropsolidConfig = $this->createMock(Config::class);
    $dropsolidConfig->method('get')->willReturnCallback(static function (string $key) use ($configValues) {
      return $configValues[$key] ?? NULL;
    });

    $liteConfig = $this->createMock(Config::class);
    $liteConfig->method('get')->willReturn(NULL);

    $configFactory = $this->createMock(ConfigFactoryInterface::class);
    $configFactory
      ->method('get')
      ->willReturnMap([
        ['ai_dropsolid.settings', $dropsolidConfig],
        ['ai_provider_litellm.settings', $liteConfig],
      ]);

    $fileSystem = $this->createMock(FileSystemInterface::class);
    $fileSystem
      ->method('realpath')
      ->willReturnArgument(0);

    $storage = $this->createMock(EntityStorageInterface::class);
    $storage->method('load')->willReturn(NULL);

    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager
      ->method('getStorage')
      ->with('file')
      ->willReturn($storage);

    $keyRepository = $this->createMock(KeyRepositoryInterface::class);
    $keyRepository->method('getKey')->willReturn(NULL);

    $httpClient = $this->createMock(ClientInterface::class);

    $logger = $this->createMock(LoggerInterface::class);

    return new DropsolidXlmRobertaTokenizer($configFactory, $fileSystem, $entityTypeManager, $keyRepository, $httpClient, $logger);
  }

}
