<?php

declare(strict_types=1);

namespace Drupal\ai_experience_wizard\Form;

use Drupal\Component\Serialization\SerializationInterface;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Recipe\Recipe;
use Drupal\Core\Recipe\RecipeRunner;
use Drupal\ai_experience_wizard\Service\AiProviderBatchOperations;
use Drupal\ai_experience_wizard\Service\AiProviderManager;
use Drupal\ai_experience_wizard\Service\KeyManager;
use Drupal\ai_experience_wizard\Service\ProviderFormPreprocessor;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a two-step wizard for AI provider selection and configuration.
 */
final class AiExperienceWizardForm extends ConfigFormBase {

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected FormBuilderInterface $formBuilder;

  /**
   * The AI provider manager service.
   *
   * @var \Drupal\ai_experience_wizard\Service\AiProviderManager
   */
  protected AiProviderManager $providerManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The key manager service.
   *
   * @var \Drupal\ai_experience_wizard\Service\KeyManager
   */
  protected KeyManager $keyManager;

  /**
   * The form preprocessor service.
   *
   * @var \Drupal\ai_experience_wizard\Service\ProviderFormPreprocessor
   */
  protected ProviderFormPreprocessor $formPreprocessor;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The app root.
   *
   * @var string
   */
  protected string $appRoot;

  /**
   * The YAML serialization service.
   *
   * @var \Drupal\Component\Serialization\SerializationInterface
   */
  protected SerializationInterface $yamlSerialization;

  /**
   * Constructs a new AiExperienceWizardForm object.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    FormBuilderInterface $form_builder,
    AiProviderManager $provider_manager,
    ModuleHandlerInterface $module_handler,
    KeyManager $key_manager,
    ProviderFormPreprocessor $form_preprocessor,
    RequestStack $request_stack,
    string $app_root,
    SerializationInterface $yaml_serialization,
  ) {
    parent::__construct($config_factory, $typed_config_manager);

    $this->formBuilder       = $form_builder;
    $this->providerManager   = $provider_manager;
    $this->moduleHandler     = $module_handler;
    $this->keyManager        = $key_manager;
    $this->formPreprocessor  = $form_preprocessor;
    $this->requestStack      = $request_stack;
    $this->appRoot           = $app_root;
    $this->yamlSerialization = $yaml_serialization;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('form_builder'),
      $container->get('ai_experience_wizard.provider_manager'),
      $container->get('module_handler'),
      $container->get('ai_experience_wizard.key_manager'),
      $container->get('ai_experience_wizard.provider_form_preprocessor'),
      $container->get('request_stack'),
      $container->getParameter('app.root'),
      $container->get('serialization.yaml')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ai_experience_wizard_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['ai_experience_wizard.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $step = $this->determineStep($form_state);
    $form_state->set('wizard_step', $step);

    $step_labels = [
      1 => $this->t('Select AI Provider'),
      2 => $this->t('Configure AI Provider'),
      3 => $this->t('Select Recipes'),
      4 => $this->t('Summary'),
    ];

    $form['wizard_header'] = [
      '#type'       => 'container',
      '#attributes' => ['class' => ['ai-experience-wizard__header']],
      '#markup'     => $this->t('Step @current/4: @label', [
        '@current' => $step,
        '@label'   => $step_labels[$step] ?? $step_labels[1],
      ]),
    ];

    if ($step === 1) {
      return $this->buildProviderStep($form, $form_state);
    }
    if ($step === 2) {
      return $this->buildConfigStep($form, $form_state);
    }
    if ($step === 3) {
      return $this->buildRecipesStep($form, $form_state);
    }

    return $this->buildSummaryStep($form, $form_state);
  }

  /**
   * Builds step 1 (provider selection).
   */
  protected function buildProviderStep(array $form, FormStateInterface $form_state): array {
    if (!$this->moduleHandler->moduleExists('package_manager')) {
      $form['package_manager_warning'] = [
        '#type'       => 'container',
        '#attributes' => ['class' => ['messages', 'messages--warning']],
        '#markup'     => $this->t('Package Manager module is not enabled. AI provider packages cannot be automatically installed. Please enable the Package Manager module.'),
      ];
    }

    $selected_provider = $this->getSelectedProvider($form_state);

    $form['ai_provider'] = [
      '#type'          => 'radios',
      '#title'         => $this->t('Select AI Provider'),
      '#description'   => $this->t('Choose your preferred AI provider for content generation and analysis.'),
      '#required'      => TRUE,
      '#options'       => $this->providerManager->getProviderOptions(),
      '#default_value' => $selected_provider ?? 'openai',
    ];

    $form['provider_descriptions'] = [
      '#type'       => 'container',
      '#attributes' => ['class' => ['ai-provider-descriptions']],
    ];

    foreach ($this->providerManager->getAllProviders() as $id => $provider) {
      $form['provider_descriptions'][$id] = [
        '#type'       => 'container',
        '#attributes' => ['class' => ['provider-description', "provider-{$id}"]],
        '#markup'     => $provider['description'],
        '#states'     => [
          'visible' => [
            ':input[name="ai_provider"]' => ['value' => $id],
          ],
        ],
      ];
    }

    $form['actions'] = [
      '#type' => 'actions',
      'next'  => [
        '#type'   => 'submit',
        '#value'  => $this->t('Next'),
        '#button_type' => 'primary',
        '#submit' => ['::submitSelectProvider'],
      ],
    ];

    return $form;
  }

  /**
   * Builds step 2 (provider configuration).
   */
  protected function buildConfigStep(array $form, FormStateInterface $form_state): array {
    $ai_provider = $this->getSelectedProvider($form_state);

    if (empty($ai_provider)) {
      $form['error'] = [
        '#type'       => 'container',
        '#attributes' => ['class' => ['messages', 'messages--error']],
        '#markup'     => $this->t('No AI provider selected. Please go back and select a provider first.'),
      ];
      $form['actions'] = [
        '#type' => 'actions',
        'back'  => [
          '#type'   => 'submit',
          '#value'  => $this->t('< Back'),
          '#submit' => ['::submitBackToSelection'],
        ],
      ];
      return $form;
    }

    if (!$this->providerManager->isValidProvider($ai_provider)) {
      $form['error'] = [
        '#type'       => 'container',
        '#attributes' => ['class' => ['messages', 'messages--error']],
        '#markup'     => $this->t('Invalid AI provider: @provider', ['@provider' => $ai_provider]),
      ];
      return $form;
    }

    $module_name = $this->providerManager->getModule($ai_provider);
    if (!$this->moduleHandler->moduleExists($module_name)) {
      $form['error'] = [
        '#type'       => 'container',
        '#attributes' => ['class' => ['messages', 'messages--error']],
        '#markup'     => $this->t('The @provider module is not available. Please ensure it was installed correctly.', [
          '@provider' => ucfirst($ai_provider),
        ]),
      ];
      $form['actions'] = [
        '#type' => 'actions',
        'back'  => [
          '#type'   => 'submit',
          '#value'  => $this->t('< Back'),
          '#submit' => ['::submitBackToSelection'],
        ],
      ];
      return $form;
    }

    $provider_form_class = $this->providerManager->getFormClass($ai_provider);
    if (empty($provider_form_class) || !class_exists($provider_form_class)) {
      $form['error'] = [
        '#type'       => 'container',
        '#attributes' => ['class' => ['messages', 'messages--error']],
        '#markup'     => $this->t('Configuration form for @provider is not available.', [
          '@provider' => ucfirst($ai_provider),
        ]),
      ];
      return $form;
    }

    // Build the embedded provider configuration subform.
    $form['provider_info'] = [
      '#type'       => 'container',
      '#attributes' => ['class' => ['provider-info']],
      '#markup'     => $this->t('Configuring <strong>@provider</strong> AI provider', [
        '@provider' => ucfirst($ai_provider),
      ]),
    ];

    $form['provider_config'] = [
      '#type'                   => 'details',
      '#title'                  => $this->t('Provider Configuration'),
      '#tree'                   => TRUE,
      '#open'                   => TRUE,
      '#process'                => [[static::class, 'processProviderSubform']],
      '#provider_form_class'    => $provider_form_class,
      '#ai_provider'            => $ai_provider,
      '#parents'                => ['provider_config'],
    ];

    $form['actions'] = [
      '#type' => 'actions',
      'back'  => [
        '#type'   => 'submit',
        '#value'  => $this->t('< Back'),
        '#submit' => ['::submitBackToSelection'],
      ],
      'submit'  => [
        '#type'        => 'submit',
        '#value'       => $this->t('Next'),
        '#button_type' => 'primary',
        '#submit'      => ['::submitSaveConfiguration'],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $step = (int) $form_state->get('wizard_step');
    if ($step === 1) {
      $this->validateProviderSelection($form_state);
      return;
    }

    if ($step === 2) {
      $ai_provider = $this->getSelectedProvider($form_state);
      if (!empty($ai_provider)) {
        $this->validateKeyFields($form, $form_state, $ai_provider);
        $this->validateProviderConfiguration($form, $form_state);
      }
    }
  }

  /**
   * Handles provider selection submission (step 1).
   */
  public function submitSelectProvider(array &$form, FormStateInterface $form_state): void {
    $ai_provider = (string) $form_state->getValue('ai_provider');
    $form_state->set('ai_provider', $ai_provider);

    $this->configFactory->getEditable('ai_experience_wizard.settings')
      ->set('ai_provider', $ai_provider)
      ->set('provider_configured', FALSE)
      ->save();

    $batch = (new BatchBuilder())
      ->setTitle($this->t('Installing AI Provider...'))
      ->setInitMessage($this->t('Starting AI provider installation...'))
      ->setProgressMessage($this->t('Processing @current of @total.'))
      ->setErrorMessage($this->t('An error occurred during installation.'))
      ->setFinishCallback([AiProviderBatchOperations::class, 'batchFinished']);

    $batch->addOperation([AiProviderBatchOperations::class, 'pullAiProvider'], [$ai_provider]);
    $batch->addOperation([AiProviderBatchOperations::class, 'installAiProvider'], [$ai_provider]);

    batch_set($batch->toArray());

    $form_state->set('wizard_step', 2);
    $form_state->setRedirect('ai_experience_wizard.wizard', [], ['query' => ['step' => 2]]);
  }

  /**
   * Handles back navigation to provider selection.
   */
  public function submitBackToSelection(array &$form, FormStateInterface $form_state): void {
    $form_state->set('wizard_step', 1);
    $form_state->setRebuild();
  }

  /**
   * Handles saving provider configuration (step 2).
   */
  public function submitSaveConfiguration(array &$form, FormStateInterface $form_state): void {
    $ai_provider = $this->getSelectedProvider($form_state);
    $provider_config = $form_state->getValue('provider_config', []);

    if ($ai_provider) {
      $this->saveProviderConfiguration($ai_provider, $provider_config, $form_state);

      $this->configFactory->getEditable('ai_experience_wizard.settings')
        ->set('provider_configured', TRUE)
        ->set('configuration_completed', time())
        ->save();

      $this->messenger()->addStatus($this->t('@provider configuration has been saved successfully.', [
        '@provider' => ucfirst($ai_provider),
      ]));
    }

    // Advance to recipes step.
    $form_state->set('wizard_step', 3);
    $form_state->setRedirect('ai_experience_wizard.wizard', [], ['query' => ['step' => 3]]);
  }

  /**
   * Ensures a provider is selected.
   */
  protected function validateProviderSelection(FormStateInterface $form_state): void {
    if (!$form_state->getValue('ai_provider')) {
      $form_state->setErrorByName('ai_provider', $this->t('Please select an AI provider.'));
    }
  }

  /**
   * Validates key fields and creates keys as needed.
   */
  protected function validateKeyFields(array &$form, FormStateInterface $form_state, string $provider): void {
    $provider_config = $form_state->getValue('provider_config', []);

    // Validate replaced fields (key_select fields).
    $replaced_fields = $form_state->getTemporaryValue('ai_experience_wizard.replaced_fields') ?? [];
    foreach ($replaced_fields as $field_name => $field_info) {
      $is_required = $field_info['required'] ?? FALSE;
      if (!$is_required) {
        continue;
      }

      $field_value = $provider_config[$field_name] ?? NULL;
      $key_value = NULL;

      if (is_array($field_value) && isset($field_value['key_input'])) {
        $key_value = trim((string) $field_value['key_input']);
      }
      elseif (is_string($field_value) && !empty($field_value)) {
        $key_value = trim($field_value);
      }

      if (empty($key_value)) {
        $form_state->setErrorByName("provider_config][{$field_name}", $this->t('@field is required.', [
          '@field' => $field_info['original_title'] ?? $field_name,
        ]));
      }
    }

    // Validate sensitive fields.
    $sensitive_fields = $form_state->getTemporaryValue('ai_experience_wizard.sensitive_fields') ?? [];
    foreach ($sensitive_fields as $field_name => $field_info) {
      $is_required = $field_info['required'] ?? FALSE;
      if (!$is_required) {
        continue;
      }

      $key_value = trim((string) ($provider_config[$field_name] ?? ''));
      if (empty($key_value)) {
        $field_title = is_object($field_info['original_title']) 
          ? $field_info['original_title']->render() 
          : (string) ($field_info['original_title'] ?? $field_name);
        $form_state->setErrorByName("provider_config][{$field_name}", $this->t('@field is required.', [
          '@field' => $field_title,
        ]));
      }
    }
  }

  /**
   * Validates provider configuration.
   */
  protected function validateProviderConfiguration(array &$form, FormStateInterface $form_state): void {
    // Additional provider-specific validation can be added here.
  }

  /**
   * Saves the provider configuration directly.
   */
  protected function saveProviderConfiguration(string $ai_provider, array $config_values, FormStateInterface $form_state): void {
    $config_name = $this->providerManager->getConfigName($ai_provider);
    if (empty($config_name)) {
      $this->logger('ai_experience_wizard')->error('Unknown provider config: @provider', [
        '@provider' => $ai_provider,
      ]);
      return;
    }

    // Process key fields and create keys.
    $processed_values = $this->processKeyFields($ai_provider, $config_values, $form_state);
    $filtered_values = $this->filterConfigurationValues($processed_values);

    $config = $this->configFactory->getEditable($config_name);
    foreach ($filtered_values as $key => $value) {
      $config->set($key, $value);
    }
    $config->save();

    $this->logger('ai_experience_wizard')->info('Saved configuration for @provider', [
      '@provider' => $ai_provider,
    ]);
  }

  /**
   * Processes key fields by creating keys and replacing values with key IDs.
   *
   * @param string $provider
   *   The AI provider name.
   * @param array $config_values
   *   The configuration values from the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The processed configuration values with key IDs instead of key values.
   */
  protected function processKeyFields(string $provider, array $config_values, FormStateInterface $form_state): array {
    $processed = $config_values;

    // Get replaced fields (key_select fields that were replaced with password inputs).
    $replaced_fields = $form_state->getTemporaryValue('ai_experience_wizard.replaced_fields') ?? [];

    // Process replaced fields (key_select -> password input).
    foreach ($replaced_fields as $field_name => $field_info) {
      // Check if this field has a value in the config.
      if (!isset($processed[$field_name])) {
        continue;
      }

      $field_value = $processed[$field_name];

      // Skip if already processed (value is already a key ID).
      if (is_string($field_value) && str_starts_with($field_value, 'ai_experience_wizard_')) {
        continue;
      }

      // Extract the key value from nested structure.
      $key_value = NULL;
      if (is_array($field_value) && isset($field_value['key_input'])) {
        $key_value = trim((string) $field_value['key_input']);
      }
      elseif (is_string($field_value) && !empty($field_value)) {
        $key_value = trim($field_value);
      }

      // Create key if we have a value.
      if (!empty($key_value)) {
        try {
          $key_id = $this->keyManager->createOrGetKey($provider, $field_name, $key_value);
          // Replace the value with the key ID.
          $processed[$field_name] = $key_id;
          $this->logger('ai_experience_wizard')->info('Created key @key_id for field @field', [
            '@key_id' => $key_id,
            '@field' => $field_name,
          ]);
        }
        catch (\Exception $e) {
          $this->logger('ai_experience_wizard')->error('Failed to create key for field @field: @error', [
            '@field' => $field_name,
            '@error' => $e->getMessage(),
          ]);
        }
      }
      else {
        // Remove the field if no value was provided.
        unset($processed[$field_name]);
      }
    }

    // Get sensitive fields (password/textfield fields marked for key creation).
    $sensitive_fields = $form_state->getTemporaryValue('ai_experience_wizard.sensitive_fields') ?? [];

    // Process sensitive fields.
    foreach ($sensitive_fields as $field_name => $field_info) {
      // Check if this field has a value in the config.
      if (!isset($processed[$field_name])) {
        continue;
      }

      // Skip if already processed (value is already a key ID).
      $field_value = $processed[$field_name];
      if (is_string($field_value) && str_starts_with($field_value, 'ai_experience_wizard_')) {
        continue;
      }

      $key_value = trim((string) $field_value);

      // Create key if we have a value.
      if (!empty($key_value)) {
        try {
          $key_id = $this->keyManager->createOrGetKey($provider, $field_name, $key_value);
          // Replace the value with the key ID.
          $processed[$field_name] = $key_id;
          $this->logger('ai_experience_wizard')->info('Created key @key_id for sensitive field @field', [
            '@key_id' => $key_id,
            '@field' => $field_name,
          ]);
        }
        catch (\Exception $e) {
          $this->logger('ai_experience_wizard')->error('Failed to create key for sensitive field @field: @error', [
            '@field' => $field_name,
            '@error' => $e->getMessage(),
          ]);
        }
      }
      else {
        // Remove the field if no value was provided.
        unset($processed[$field_name]);
      }
    }

    return $processed;
  }

  /**
   * Filters configuration values to only include top-level values.
   */
  protected function filterConfigurationValues(array $config_values): array {
    $filtered = [];

    foreach ($config_values as $key => $value) {
      if (is_string($value)) {
        $filtered[$key] = $value;
        continue;
      }

      if (is_array($value)) {
        if (isset($value['key_input'])) {
          $this->logger('ai_experience_wizard')->warning('Found unreplaced field @key with key_input', [
            '@key' => $key,
          ]);
          continue;
        }

        if (count($value) === 1 && !isset($value['key_info']) && !isset($value['existing_keys'])) {
          $filtered[$key] = reset($value);
        }
        continue;
      }

      $filtered[$key] = $value;
    }

    return $filtered;
  }

  /**
   * Determines the active wizard step.
   */
  protected function determineStep(FormStateInterface $form_state): int {
    if ($form_state->has('wizard_step')) {
      return (int) $form_state->get('wizard_step');
    }

    $request = $this->requestStack->getCurrentRequest();
    $step = (int) $request->query->get('step', 1);
    if ($step < 1 || $step > 4) {
      $step = 1;
    }
    return $step;
  }

  /**
   * Builds step 4 (summary of configured provider and recipes).
   */
  protected function buildSummaryStep(array $form, FormStateInterface $form_state): array {
    $config = $this->config('ai_experience_wizard.settings');
    $ai_provider = $this->getSelectedProvider($form_state);
    $provider_label = $ai_provider;

    if ($ai_provider && $this->providerManager->isValidProvider($ai_provider)) {
      $info = $this->providerManager->getProviderInfo($ai_provider);
      if (!empty($info['label'])) {
        $provider_label = $info['label'];
      }
    }

    $form['summary_intro'] = [
      '#type'  => 'markup',
      '#markup' => $this->t('Your AI experience setup is complete. Here is a summary of what was configured.'),
    ];

    $form['provider_summary'] = [
      '#type' => 'details',
      '#title' => $this->t('AI provider'),
      '#open' => TRUE,
      'provider' => [
        '#type' => 'item',
        '#title' => $this->t('Selected provider'),
        '#markup' => $provider_label ? $this->t('@provider', ['@provider' => $provider_label]) : $this->t('None selected'),
      ],
    ];

    $last_applied = $config->get('last_applied_recipes') ?? [];

    $form['recipes_summary'] = [
      '#type' => 'details',
      '#title' => $this->t('Recipes'),
      '#open' => TRUE,
    ];

    if (empty($last_applied)) {
      $form['recipes_summary']['none'] = [
        '#type' => 'item',
        '#markup' => $this->t('No recipes were installed in the last run of this wizard.'),
      ];
    }
    else {
      $items = [];
      foreach ($last_applied as $id => $label) {
        $items[] = $label ? $this->t('@label (@id)', ['@label' => $label, '@id' => $id]) : $this->t('@id', ['@id' => $id]);
      }

      $form['recipes_summary']['list'] = [
        '#theme' => 'item_list',
        '#items' => $items,
      ];
    }

    $form['actions'] = [
      '#type' => 'actions',
      'finish' => [
        '#type' => 'submit',
        '#value' => $this->t('Done'),
        '#button_type' => 'primary',
        '#submit' => ['::submitFinishSummary'],
      ],
    ];

    return $form;
  }

  /**
   * Processes the embedded provider configuration subform.
   */
  public static function processProviderSubform(array &$element, FormStateInterface $form_state, array &$complete_form): array {
    /** @var \Drupal\ai_experience_wizard\Form\AiExperienceWizardForm $form_object */
    $form_object = $form_state->getFormObject();
    $provider_manager = $form_object->providerManager;
    $form_preprocessor = $form_object->formPreprocessor;
    $ai_provider = $element['#ai_provider'] ?? $form_object->getSelectedProvider($form_state);

    if (empty($ai_provider)) {
      return $element;
    }

    $provider_form_class = $element['#provider_form_class']
      ?? $provider_manager->getFormClass($ai_provider);
    if (empty($provider_form_class) || !class_exists($provider_form_class)) {
      return $element;
    }

    /** @var \Drupal\Core\Form\FormBuilderInterface $form_builder */
    $form_builder = \Drupal::service('form_builder');
    $provider_form = $form_builder->getForm($provider_form_class);

    // Preprocess the provider form to attach element-level validation.
    $provider_form = $form_preprocessor->preprocessForm($provider_form, $form_state, $ai_provider);

    // Merge provider form elements into this subform container, skipping
    // top-level form infrastructure and actions so we don't end up with a
    // nested form (which can cause outdated form errors) or duplicate
    // submit buttons.
    $skip_keys = [
      'actions',
      'form_id',
      'form_build_id',
      'form_token',
      '#token',
      '#build_id',
      '#method',
      '#id',
    ];

    foreach ($provider_form as $key => $child) {
      if (in_array($key, $skip_keys, TRUE)) {
        continue;
      }
      if (str_starts_with((string) $key, '#')) {
        continue;
      }
      $element[$key] = $child;
    }

    return $element;
  }

  /**
   * Builds step 3 (recipes selection and applying).
   */
  protected function buildRecipesStep(array $form, FormStateInterface $form_state): array {
    $form['recipes_intro'] = [
      '#type' => 'markup',
      '#markup' => $this->t('Select one or more site recipes to apply. These can include core and contributed recipes discovered on your site.'),
    ];

    $recipes = $this->discoverRecipes();

    if (empty($recipes)) {
      $form['no_recipes'] = [
        '#type' => 'markup',
        '#markup' => $this->t('No recipes were found in the standard recipe locations.'),
      ];
    }
    else {
      $options = [];
      foreach ($recipes as $id => $info) {
        $label = $info['label'] ?? $id;
        $source = $info['source'] ?? 'unknown';
        $options[$id] = $this->t('@label (@source)', [
          '@label' => $label,
          '@source' => $source,
        ]);
      }

      $form['recipes'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Available recipes'),
        '#options' => $options,
        '#description' => $this->t('Select the recipes you would like to install.'),
      ];
    }

    $form['actions'] = [
      '#type' => 'actions',
      'back' => [
        '#type' => 'submit',
        '#value' => $this->t('< Back'),
        '#submit' => ['::submitBackToConfig'],
      ],
      'finish' => [
        '#type' => 'submit',
        '#value' => $this->t('Install selected recipes & Finish'),
        '#button_type' => 'primary',
        '#submit' => ['::submitInstallRecipes'],
      ],
    ];

    return $form;
  }

  /**
   * Handles back navigation from recipes to provider configuration.
   */
  public function submitBackToConfig(array &$form, FormStateInterface $form_state): void {
    $form_state->set('wizard_step', 2);
    $form_state->setRedirect('ai_experience_wizard.wizard', [], ['query' => ['step' => 2]]);
  }

  /**
   * Handles installation of selected recipes.
   */
  public function submitInstallRecipes(array &$form, FormStateInterface $form_state): void {
    $selected = array_filter($form_state->getValue('recipes', []));
    $applied_info = [];

    if ($selected) {
      $available = $this->discoverRecipes();
      $applied = [];
      $failed = [];

      foreach ($selected as $id) {
        if (!isset($available[$id])) {
          $failed[] = $id;
          continue;
        }

        $path = $available[$id]['path'] ?? NULL;
        $label = $available[$id]['label'] ?? $id;
        if (!$path || !is_dir($path)) {
          $failed[] = $id;
          continue;
        }

        try {
          $recipe = Recipe::createFromDirectory($path);
          RecipeRunner::processRecipe($recipe);
          $applied[] = $id;
          $applied_info[$id] = $label;
        }
        catch (\Throwable $e) {
          $failed[] = $id;
          $this->logger('ai_experience_wizard')->error('Failed to apply recipe @id: @message', [
            '@id' => $id,
            '@message' => $e->getMessage(),
          ]);
        }
      }

      if ($applied) {
        $this->messenger()->addStatus($this->t('Applied recipes: @recipes', [
          '@recipes' => implode(', ', $applied),
        ]));

        $this->configFactory->getEditable('ai_experience_wizard.settings')
          ->set('last_applied_recipes', $applied_info)
          ->save();
      }

      if ($failed) {
        $this->messenger()->addError($this->t('Some recipes could not be applied: @recipes. Check logs for details.', [
          '@recipes' => implode(', ', $failed),
        ]));
      }
    }

    if (empty($applied_info)) {
      $this->configFactory->getEditable('ai_experience_wizard.settings')
        ->clear('last_applied_recipes')
        ->save();
    }

    $form_state->set('wizard_step', 4);
    $form_state->setRedirect('ai_experience_wizard.wizard', [], ['query' => ['step' => 4]]);
  }

  /**
   * Discovers available recipes from core and contrib locations.
   *
   * @return array
   *   An array of recipe info keyed by machine name.
   */
  protected function discoverRecipes(): array {
    $recipes = [];
    $root = $this->appRoot;
    $project_root = dirname($root);

    $paths = [
      $root . '/core/recipes',
      $project_root . '/recipes',
    ];

    foreach ($paths as $path) {
      if (!is_dir($path)) {
        continue;
      }

      $dir = scandir($path) ?: [];
      foreach ($dir as $entry) {
        if ($entry === '.' || $entry === '..') {
          continue;
        }
        $recipe_dir = $path . '/' . $entry;
        if (!is_dir($recipe_dir)) {
          continue;
        }

        // Support both core-style "recipe.yml" and custom "<name>.recipe.yml".
        $candidate_files = [
          $recipe_dir . '/' . $entry . '.recipe.yml',
          $recipe_dir . '/recipe.yml',
        ];

        $info_file = NULL;
        foreach ($candidate_files as $candidate) {
          if (is_file($candidate)) {
            $info_file = $candidate;
            break;
          }
        }

        if ($info_file) {
          try {
            $data = $this->yamlSerialization->decode(file_get_contents($info_file)) ?? [];
          }
          catch (\Throwable $e) {
            $data = [];
          }

          $recipes[$entry] = [
            'label' => $data['name'] ?? $entry,
            'description' => $data['description'] ?? '',
            'source' => str_contains($path, '/core/') ? 'core' : 'contrib',
            'path' => $recipe_dir,
          ];
        }
      }
    }

    return $recipes;
  }

  /**
   * Retrieves the selected provider from state or config.
   */
  protected function getSelectedProvider(FormStateInterface $form_state): ?string {
    if ($form_state->has('ai_provider')) {
      return (string) $form_state->get('ai_provider');
    }

    $value = $form_state->getValue('ai_provider');
    if (!empty($value)) {
      return (string) $value;
    }

    $config_value = $this->config('ai_experience_wizard.settings')->get('ai_provider');
    return $config_value ? (string) $config_value : NULL;
  }

  /**
   * Handles completion of the summary step.
   */
  public function submitFinishSummary(array &$form, FormStateInterface $form_state): void {
    $form_state->setRedirect('<front>');
  }

}
