<?php

namespace Drupal\ai_experience_wizard\Service;

use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\ai_experience_wizard\Service\AiSandboxPackageManagerInstaller;
use Drupal\ai_experience_wizard\Service\AiProviderManager;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;

/**
 * Service for installing AI provider packages using Package Manager.
 */
class AiProviderInstaller implements LoggerAwareInterface {

  use LoggerAwareTrait;
  use StringTranslationTrait;

  /**
   * The module extension list service.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected $moduleExtensionList;

  /**
   * The AI sandbox package manager installer service.
   *
   * @var \Drupal\ai_experience_wizard\Service\AiSandboxPackageManagerInstaller
   */
  protected $directInstaller;

  /**
   * The AI provider manager service.
   *
   * @var \Drupal\ai_experience_wizard\Service\AiProviderManager
   */
  protected $providerManager;

  /**
   * Constructs an AiProviderInstaller object.
   *
   * @param \Drupal\Core\Extension\ModuleExtensionList $module_extension_list
   *   The module extension list service.
   * @param \Drupal\ai_experience_wizard\Service\AiSandboxPackageManagerInstaller $direct_installer
   *   The AI sandbox package manager installer service.
   * @param \Drupal\ai_experience_wizard\Service\AiProviderManager $provider_manager
   *   The AI provider manager service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(
    ModuleExtensionList $module_extension_list,
    AiSandboxPackageManagerInstaller $direct_installer,
    AiProviderManager $provider_manager,
    LoggerInterface $logger
  ) {
    $this->moduleExtensionList = $module_extension_list;
    $this->directInstaller = $direct_installer;
    $this->providerManager = $provider_manager;
    $this->setLogger($logger);
  }

  /**
   * Installs the required AI provider package.
   *
   * @param string $provider
   *   The AI provider name.
   *
   * @return bool
   *   TRUE if the package was installed successfully, FALSE otherwise.
   */
  public function requireProviderOnly(string $provider): bool {
    // Validate provider using the registry
    if (!$this->providerManager->isValidProvider($provider)) {
      $this->logger->warning('Unknown AI provider: @provider', ['@provider' => $provider]);
      return FALSE;
    }

    $package = $this->providerManager->getPackage($provider);
    $module = $this->providerManager->getModule($provider);
    $available_modules = array_keys($this->moduleExtensionList->getList());

    if (in_array($module, $available_modules, TRUE)) {
      $this->logger->info('AI provider module @module is already available.', ['@module' => $module]);
      return TRUE;
    }

    // First, ensure the base AI module is installed
    if (!$this->ensureBaseAiModule()) {
      $this->logger->error('Failed to install base AI module');
      return FALSE;
    }

    // Use the direct Package Manager installer
    if ($this->directInstaller->installPackage($package)) {
      // Clear module cache to ensure new modules are discovered
      $this->moduleExtensionList->reset();
      $this->logger->info('Successfully installed AI provider package: @package', ['@package' => $package]);
      return TRUE;
    } else {
      $this->logger->error('Failed to install AI provider package: @package', ['@package' => $package]);
      return FALSE;
    }
  }

  /**
   * Ensures the base AI module is installed.
   *
   * @return bool
   *   TRUE if the base AI module is available, FALSE otherwise.
   */
  protected function ensureBaseAiModule(): bool {
    $available_modules = array_keys($this->moduleExtensionList->getList());

    // Check if base AI module is already available
    if (in_array('ai', $available_modules, TRUE)) {
      $this->logger->info('Base AI module is already available.');
      return TRUE;
    }

    // Install the base AI module first
    if ($this->directInstaller->installPackage('drupal/ai')) {
      $this->logger->info('Successfully installed base AI module.');
      return TRUE;
    } else {
      $this->logger->error('Failed to install base AI module.');
      return FALSE;
    }
  }

  /**
   * Checks if a specific AI provider package is already installed.
   *
   * @param string $provider
   *   The AI provider name.
   *
   * @return bool
   *   TRUE if the package is already installed, FALSE otherwise.
   */
  public function isPackageInstalled(string $provider): bool {
    if (!$this->providerManager->isValidProvider($provider)) {
      return FALSE;
    }

    $package = $this->providerManager->getPackage($provider);

    // Use the direct installer to check package status
    return $this->directInstaller->isPackageAlreadyInstalled($package);
  }

  /**
   * Checks if a specific AI provider module is already installed.
   *
   * @param string $provider
   *   The AI provider name.
   *
   * @return bool
   *   TRUE if the module is already installed, FALSE otherwise.
   */
  public function isProviderInstalled(string $provider): bool {
    if (!$this->providerManager->isValidProvider($provider)) {
      return FALSE;
    }

    $module = $this->providerManager->getModule($provider);
    $available_modules = array_keys($this->moduleExtensionList->getList());

    return in_array($module, $available_modules, TRUE);
  }

}
