<?php

declare(strict_types=1);

namespace Drupal\ai_experience_wizard\Service;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\ai_experience_wizard\Config\AiProviderRegistry;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;

/**
 * Service for managing AI providers.
 *
 * This service provides a centralized way to manage AI providers,
 * making it easy to add new providers by updating the registry.
 */
final class AiProviderManager implements LoggerAwareInterface {

  use LoggerAwareTrait;
  use StringTranslationTrait;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs an AiProviderManager object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   */
  public function __construct(ModuleHandlerInterface $module_handler) {
    $this->moduleHandler = $module_handler;
  }

  /**
   * Gets all available providers.
   *
   * @return array
   *   Array of provider definitions.
   */
  public function getAllProviders(): array {
    return AiProviderRegistry::getAllProviders();
  }

  /**
   * Gets provider options for form elements.
   *
   * @return array
   *   Array of provider options.
   */
  public function getProviderOptions(): array {
    $options = [];
    foreach ($this->getAllProviders() as $id => $provider) {
      $options[$id] = $provider['label'];
    }
    return $options;
  }

  /**
   * Gets a specific provider definition.
   *
   * @param string $provider_id
   *   The provider ID.
   *
   * @return array|null
   *   The provider definition or NULL if not found.
   */
  public function getProvider(string $provider_id): ?array {
    return AiProviderRegistry::getProvider($provider_id);
  }

  /**
   * Checks if a provider is valid.
   *
   * @param string $provider_id
   *   The provider ID.
   *
   * @return bool
   *   TRUE if valid, FALSE otherwise.
   */
  public function isValidProvider(string $provider_id): bool {
    return AiProviderRegistry::isValidProvider($provider_id);
  }

  /**
   * Gets package name for a provider.
   *
   * @param string $provider_id
   *   The provider ID.
   *
   * @return string|null
   *   The package name or NULL if not found.
   */
  public function getPackage(string $provider_id): ?string {
    return AiProviderRegistry::getPackage($provider_id);
  }

  /**
   * Gets module name for a provider.
   *
   * @param string $provider_id
   *   The provider ID.
   *
   * @return string|null
   *   The module name or NULL if not found.
   */
  public function getModule(string $provider_id): ?string {
    return AiProviderRegistry::getModule($provider_id);
  }

  /**
   * Gets config name for a provider.
   *
   * @param string $provider_id
   *   The provider ID.
   *
   * @return string|null
   *   The config name or NULL if not found.
   */
  public function getConfigName(string $provider_id): ?string {
    return AiProviderRegistry::getConfigName($provider_id);
  }

  /**
   * Gets form class for a provider.
   *
   * @param string $provider_id
   *   The provider ID.
   *
   * @return string|null
   *   The form class or NULL if not found.
   */
  public function getFormClass(string $provider_id): ?string {
    return AiProviderRegistry::getFormClass($provider_id);
  }

  /**
   * Checks if a provider module is installed.
   *
   * @param string $provider_id
   *   The provider ID.
   *
   * @return bool
   *   TRUE if installed, FALSE otherwise.
   */
  public function isProviderInstalled(string $provider_id): bool {
    $module = $this->getModule($provider_id);
    if (!$module) {
      return FALSE;
    }

    return $this->moduleHandler->moduleExists($module);
  }

  /**
   * Gets provider information for display.
   *
   * @param string $provider_id
   *   The provider ID.
   *
   * @return array|null
   *   Provider information array or NULL if not found.
   */
  public function getProviderInfo(string $provider_id): ?array {
    $provider = $this->getProvider($provider_id);
    if (!$provider) {
      return NULL;
    }

    return [
      'id' => $provider_id,
      'label' => $provider['label'],
      'description' => $provider['description'],
      'website' => $provider['website'],
      'icon' => $provider['icon'],
      'installed' => $this->isProviderInstalled($provider_id),
      'package' => $provider['package'],
      'module' => $provider['module'],
    ];
  }

}
