<?php

namespace Drupal\ai_experience_wizard\Service;

use Drupal\package_manager\Attribute\AllowDirectWrite;
use Drupal\package_manager\ComposerInspector;
use Drupal\package_manager\Exception\SandboxException;
use Drupal\package_manager\SandboxManagerBase;
use PhpTuf\ComposerStager\API\Core\BeginnerInterface;
use PhpTuf\ComposerStager\API\Core\CommitterInterface;
use PhpTuf\ComposerStager\API\Core\StagerInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\TempStore\SharedTempStoreFactory;
use Drupal\Component\Datetime\TimeInterface;
use PhpTuf\ComposerStager\API\Path\Factory\PathFactoryInterface;
use Drupal\package_manager\FailureMarker;
use Drupal\package_manager\PathLocator;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * AI Sandbox Package Manager installer for AI provider packages.
 *
 * This service extends SandboxManagerBase to provide direct access to
 * Package Manager functionality without depending on Project Browser.
 */
#[AllowDirectWrite]
class AiSandboxPackageManagerInstaller extends SandboxManagerBase {

  /**
   * {@inheritdoc}
   */
  protected string $type = 'ai_experience_wizard.installer';

  /**
   * The Composer inspector.
   *
   * @var \Drupal\package_manager\ComposerInspector
   */
  protected ComposerInspector $composerInspector;

  /**
   * Constructs a new AiSandboxPackageManagerInstaller object.
   *
   * @param \Drupal\package_manager\PathLocator $path_locator
   *   The path locator.
   * @param \PhpTuf\ComposerStager\API\Core\BeginnerInterface $beginner
   *   The beginner.
   * @param \PhpTuf\ComposerStager\API\Core\StagerInterface $stager
   *   The stager.
   * @param \PhpTuf\ComposerStager\API\Core\CommitterInterface $committer
   *   The committer.
   * @param \Drupal\Core\Queue\QueueFactory $queue_factory
   *   The queue factory.
   * @param \Symfony\Contracts\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   The event dispatcher.
   * @param \Drupal\Core\TempStore\SharedTempStoreFactory $temp_store_factory
   *   The temp store factory.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \PhpTuf\ComposerStager\API\Path\Factory\PathFactoryInterface $path_factory
   *   The path factory.
   * @param \Drupal\package_manager\FailureMarker $failure_marker
   *   The failure marker.
   * @param \Drupal\package_manager\ComposerInspector $composer_inspector
   *   The Composer inspector.
   */
  public function __construct(
    PathLocator $path_locator,
    BeginnerInterface $beginner,
    StagerInterface $stager,
    CommitterInterface $committer,
    QueueFactory $queue_factory,
    EventDispatcherInterface $event_dispatcher,
    SharedTempStoreFactory $temp_store_factory,
    TimeInterface $time,
    PathFactoryInterface $path_factory,
    FailureMarker $failure_marker,
    ComposerInspector $composer_inspector
  ) {
    parent::__construct(
      $path_locator,
      $beginner,
      $stager,
      $committer,
      $queue_factory,
      $event_dispatcher,
      $temp_store_factory,
      $time,
      $path_factory,
      $failure_marker
    );
    $this->composerInspector = $composer_inspector;
  }

  /**
   * Installs a package using Package Manager directly.
   *
   * @param string $package
   *   The package name to install (e.g., 'drupal/ai_provider_openai').
   *
   * @return bool
   *   TRUE if the package was installed successfully, FALSE otherwise.
   */
  public function installPackage(string $package): bool {
    $sandbox_created = FALSE;
    
    try {
      // Check if package is already installed
      if ($this->isPackageAlreadyInstalled($package)) {
        if ($this->logger) {
          $this->logger->info('Package @package is already installed, skipping installation.', [
            '@package' => $package,
          ]);
        }
        return TRUE;
      }

      // Create a sandbox (will throw SandboxException if one already exists)
      // Note: create() automatically claims the sandbox, so no need to call claim()
      $sandbox_id = $this->create();
      $sandbox_created = TRUE;
      
      // Require the package with specific version constraints to avoid conflicts
      $this->requirePackageWithConstraints($package);
      
      // Apply the changes
      $this->apply();
      
      // Post-apply cleanup
      $this->postApply();
      
      // Destroy the sandbox on success
      $this->destroy();
      $sandbox_created = FALSE;
      
      return TRUE;
    }
    catch (SandboxException $e) {
      // Handle sandbox exceptions (e.g., sandbox already exists)
      if ($this->logger) {
        $this->logger->warning('Sandbox exception for package @package: @message', [
          '@package' => $package,
          '@message' => $e->getMessage(),
        ]);
      }

      // Check if this is because another update is in progress
      if (strpos($e->getMessage(), 'already exists') !== FALSE || strpos($e->getMessage(), 'already been created') !== FALSE) {
        if ($this->logger) {
          $this->logger->error('Cannot install package @package: Another package update operation is already in progress. Please wait for it to complete before starting a new installation.', [
            '@package' => $package,
          ]);
        }
        return FALSE;
      }

      // Re-throw other sandbox exceptions
      throw $e;
    }
    catch (\Drupal\package_manager\Exception\SandboxEventException $e) {
      // Handle sandbox event exceptions specifically
      if ($this->logger) {
        $this->logger->warning('Sandbox event exception for package @package: @message', [
          '@package' => $package,
          '@message' => $e->getMessage(),
        ]);
      }
      
      // Check if this is a "no pending operations" issue (package already installed)
      if (strpos($e->getMessage(), 'no pending Composer operations') !== FALSE) {
        if ($this->logger) {
          $this->logger->info('No pending Composer operations detected for package @package. Package may already be installed.', [
            '@package' => $package,
          ]);
        }
        // Return TRUE since the package is likely already installed
        return TRUE;
      }
      
      // Check if this is a version conflict issue
      if (strpos($e->getMessage(), 'not in the list of installable releases') !== FALSE) {
        if ($this->logger) {
          $this->logger->error('Version conflict detected for package @package. This may be due to incompatible package versions or missing dependencies.', [
            '@package' => $package,
          ]);
        }
        return FALSE;
      }
      
      // For other sandbox event exceptions, check if the package is already installed
      return $this->isPackageAlreadyInstalled($package);
    }
    catch (\Throwable $e) {
      if ($this->logger) {
        $this->logger->error('Failed to install package @package: @message', [
          '@package' => $package,
          '@message' => $e->getMessage(),
        ]);
      }
      
      return FALSE;
    }
    finally {
      // Ensure cleanup of sandbox we created, regardless of how we exit
      if ($sandbox_created) {
        try {
          // Only destroy if we own the sandbox (i.e., we created it)
          if (!$this->isAvailable()) {
            $this->destroy();
          }
        }
        catch (\Throwable $destroy_exception) {
          // Log but don't throw - the main error is more important
          if ($this->logger) {
            $this->logger->warning('Failed to destroy sandbox in finally block: @message', [
              '@message' => $destroy_exception->getMessage(),
            ]);
          }
        }
      }
    }
  }

  /**
   * Requires a package without version constraints to let Composer handle version resolution.
   *
   * @param string $package
   *   The package name to install.
   */
  protected function requirePackageWithConstraints(string $package): void {
    // Require the package without version constraints to avoid conflicts
    // Let Composer handle version resolution automatically
    $this->require([$package]);
  }


  /**
   * Checks if a package is already installed.
   *
   * @param string $package
   *   The package name to check.
   *
   * @return bool
   *   TRUE if the package is already installed, FALSE otherwise.
   */
  public function isPackageAlreadyInstalled(string $package): bool {
    try {
      // Use ComposerInspector to check if package is already installed
      $installed_packages = $this->composerInspector->getInstalledPackages();
      
      foreach ($installed_packages as $installed_package) {
        if ($installed_package->name === $package) {
          return TRUE;
        }
      }
      
      return FALSE;
    }
    catch (\Throwable $e) {
      // If we can't check, assume it's not installed
      if ($this->logger) {
        $this->logger->warning('Could not check if package @package is installed: @message', [
          '@package' => $package,
          '@message' => $e->getMessage(),
        ]);
      }
      return FALSE;
    }
  }

}
