<?php

declare(strict_types=1);

namespace Drupal\ai_experience_wizard\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\key\KeyRepositoryInterface;
use Drupal\Core\Render\Element;

/**
 * Service for preprocessing AI provider forms to replace key_select fields.
 */
final class ProviderFormPreprocessor {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * Constructs a new ProviderFormPreprocessor object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, KeyRepositoryInterface $key_repository) {
    $this->entityTypeManager = $entity_type_manager;
    $this->keyRepository = $key_repository;
  }

  /**
   * Preprocesses a provider form to replace key_select and sensitive fields.
   *
   * @param array $form
   *   The form array to preprocess.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $provider
   *   The AI provider name.
   *
   * @return array
   *   The preprocessed form array.
   */
  public function preprocessForm(array $form, FormStateInterface $form_state, string $provider): array {
    $this->preprocessFormRecursive($form, $form_state, $provider);
    return $form;
  }

  /**
   * Recursively preprocesses form elements.
   *
   * @param array &$form
   *   The form array to preprocess.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $provider
   *   The AI provider name.
   */
  protected function preprocessFormRecursive(array &$form, FormStateInterface $form_state, string $provider): void {
    foreach (Element::children($form) as $key) {
      $element = &$form[$key];

      // Skip actions and submit buttons.
      if (in_array($element['#type'] ?? '', ['actions', 'submit', 'button'])) {
        continue;
      }

      // Process key_select fields.
      if (($element['#type'] ?? '') === 'key_select') {
        $this->replaceKeySelectField($element, $form_state, $provider, $key);
      }
      // Process password fields.
      elseif (($element['#type'] ?? '') === 'password') {
        $this->markFieldForKeyCreation($element, $form_state, $provider, $key);
      }
      // Process textfield fields with sensitive names.
      elseif (($element['#type'] ?? '') === 'textfield' && $this->isSensitiveField($key)) {
        $this->markFieldForKeyCreation($element, $form_state, $provider, $key);
      }

      // Recursively process child elements.
      if (is_array($element)) {
        $this->preprocessFormRecursive($element, $form_state, $provider);
      }
    }
  }

  /**
   * Replaces a key_select field with a password input and info container.
   *
   * @param array &$element
   *   The form element to replace.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $provider
   *   The AI provider name.
   * @param string $field_name
   *   The field name.
   */
  protected function replaceKeySelectField(array &$element, FormStateInterface $form_state, string $provider, string $field_name): void {
    $original_title = $element['#title'] ?? '';
    $original_description = $element['#description'] ?? '';
    $original_required = $element['#required'] ?? FALSE;

    // Ensure we have proper strings.
    $original_title = is_object($original_title) ? $original_title->render() : (string) $original_title;
    $original_description = is_object($original_description) ? $original_description->render() : (string) $original_description;

    // Store original field info for later use.
    $replaced_fields = $form_state->getTemporaryValue('ai_experience_wizard.replaced_fields') ?? [];
    $replaced_fields[$field_name] = [
      'type' => 'key_select',
      'provider' => $provider,
      'original_title' => $original_title,
      'original_description' => $original_description,
    ];
    $form_state->setTemporaryValue('ai_experience_wizard.replaced_fields', $replaced_fields);

    // Create a container to hold the replacement elements.
    $element = [
      '#type' => 'container',
      '#attributes' => ['class' => ['ai-experience-wizard-key-replacement']],
    ];

    // Add info about automatic key creation.
    $element['key_info'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['messages', 'messages--info']],
      '#markup' => $this->t('A new key will be created automatically with the name: <strong>@key_name</strong>', [
        '@key_name' => "ai_experience_wizard_{$provider}_{$field_name}",
      ])->render(),
    ];

    // Add the password input field.
    $element['key_input'] = [
      '#type' => 'password',
      '#title' => $original_title,
      '#description' => $original_description,
      '#required' => $original_required,
      '#attributes' => [
        'placeholder' => $this->t('Enter your @title', ['@title' => strtolower($original_title)]),
      ],
    ];

    // Store the required state for validation.
    $replaced_fields = $form_state->getTemporaryValue('ai_experience_wizard.replaced_fields') ?? [];
    if (!isset($replaced_fields[$field_name])) {
      $replaced_fields[$field_name] = [];
    }
    $replaced_fields[$field_name]['required'] = $original_required;
    $form_state->setTemporaryValue('ai_experience_wizard.replaced_fields', $replaced_fields);

    // Show existing keys if any.
    $existing_keys = $this->getExistingKeys($provider, $field_name);
    if (!empty($existing_keys)) {
      $element['existing_keys'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['existing-keys-info']],
        '#markup' => $this->t('Existing keys: @keys', [
          '@keys' => implode(', ', array_keys($existing_keys)),
        ])->render(),
      ];
    }
  }

  /**
   * Marks a field for key creation during validation.
   *
   * @param array &$element
   *   The form element to mark.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $provider
   *   The AI provider name.
   * @param string $field_name
   *   The field name.
   */
  protected function markFieldForKeyCreation(array &$element, FormStateInterface $form_state, string $provider, string $field_name): void {
    // Store field info for validation.
    $sensitive_fields = $form_state->getTemporaryValue('ai_experience_wizard.sensitive_fields') ?? [];
    $sensitive_fields[$field_name] = [
      'type' => $element['#type'],
      'provider' => $provider,
      'original_title' => $element['#title'] ?? '',
      'original_description' => $element['#description'] ?? '',
    ];
    $form_state->setTemporaryValue('ai_experience_wizard.sensitive_fields', $sensitive_fields);

    // Add a note about automatic key creation.
    if (!empty($element['#description'])) {
      $element['#description'] .= ' ';
    }
    $element['#description'] .= $this->t('This value will be stored securely as a Key entity.');
  }

  /**
   * Checks if a field name indicates it contains sensitive information.
   *
   * @param string $field_name
   *   The field name to check.
   *
   * @return bool
   *   TRUE if the field is sensitive, FALSE otherwise.
   */
  protected function isSensitiveField(string $field_name): bool {
    $sensitive_patterns = ['api', 'key', 'token', 'secret', 'password', 'auth'];
    $field_lower = strtolower($field_name);

    foreach ($sensitive_patterns as $pattern) {
      if (strpos($field_lower, $pattern) !== FALSE) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Gets existing keys for a provider and field.
   *
   * @param string $provider
   *   The AI provider name.
   * @param string $field_name
   *   The field name.
   *
   * @return array
   *   Array of existing key options.
   */
  protected function getExistingKeys(string $provider, string $field_name): array {
    $key_id = "ai_experience_wizard_{$provider}_{$field_name}";
    $existing_key = $this->keyRepository->getKey($key_id);

    if ($existing_key) {
      return [$key_id => $existing_key->label()];
    }

    return [];
  }

}
