<?php

declare(strict_types=1);

namespace Drupal\ai_experience_wizard\Config;

/**
 * Registry for AI provider configurations.
 *
 * This class centralizes all AI provider definitions to make it easy
 * for developers to add new providers by simply adding entries here.
 */
final class AiProviderRegistry {

  /**
   * AI Provider definitions.
   *
   * To add a new provider, simply add a new entry to this array.
   *
   * @var array
   */
  public const PROVIDERS = [
    'openai' => [
      'label' => 'OpenAI',
      'package' => 'drupal/ai_provider_openai',
      'module' => 'ai_provider_openai',
      'config_name' => 'ai_provider_openai.settings',
      'form_class' => 'Drupal\ai_provider_openai\Form\OpenAiConfigForm',
      'description' => 'OpenAI GPT models for text generation and analysis',
      'icon' => 'openai',
      'website' => 'https://openai.com',
    ],
    'anthropic' => [
      'label' => 'Anthropic',
      'package' => 'drupal/ai_provider_anthropic',
      'module' => 'ai_provider_anthropic',
      'config_name' => 'ai_provider_anthropic.settings',
      'form_class' => 'Drupal\ai_provider_anthropic\Form\AnthropicConfigForm',
      'description' => 'Anthropic Claude models for advanced AI capabilities',
      'icon' => 'anthropic',
      'website' => 'https://anthropic.com',
    ],
    'litellm' => [
      'label' => 'LiteLLM',
      'package' => 'drupal/ai_provider_litellm',
      'module' => 'ai_provider_litellm',
      'config_name' => 'ai_provider_litellm.settings',
      'form_class' => 'Drupal\ai_provider_litellm\Form\LiteLlmAiConfigForm',
      'description' => 'LiteLLM for unified access to multiple AI providers',
      'icon' => 'litellm',
      'website' => 'https://litellm.ai',
    ],
    'groq' => [
      'label' => 'Groq',
      'package' => 'drupal/ai_provider_groq',
      'module' => 'ai_provider_groq',
      'config_name' => 'ai_provider_groq.settings',
      'form_class' => 'Drupal\ai_provider_groq\Form\GroqConfigForm',
      'description' => 'This enables the use of Groq for the AI module.',
      'icon' => 'groq',
      'website' => 'https://groq.com',
    ],

    // Add new providers here following the same pattern:
    // 'new_provider' => [
    //   'label' => 'New Provider',
    //   'package' => 'drupal/ai_provider_new_provider',
    //   'module' => 'ai_provider_new_provider',
    //   'config_name' => 'ai_provider_new_provider.settings',
    //   'form_class' => 'Drupal\ai_provider_new_provider\Form\NewProviderConfigForm',
    //   'description' => 'Description of the new provider',
    //   'icon' => 'new_provider',
    //   'website' => 'https://newprovider.com',
    // ],
  ];

  /**
   * Gets all available providers.
   *
   * @return array
   *   Array of provider definitions.
   */
  public static function getAllProviders(): array {
    return self::PROVIDERS;
  }

  /**
   * Gets a specific provider definition.
   *
   * @param string $provider_id
   *   The provider ID.
   *
   * @return array|null
   *   The provider definition or NULL if not found.
   */
  public static function getProvider(string $provider_id): ?array {
    return self::PROVIDERS[$provider_id] ?? NULL;
  }

  /**
   * Gets provider options for form elements.
   *
   * @return array
   *   Array of provider options keyed by provider ID.
   */
  public static function getProviderOptions(): array {
    $options = [];
    foreach (self::PROVIDERS as $id => $provider) {
      $options[$id] = $provider['label'];
    }
    return $options;
  }

  /**
   * Gets package name for a provider.
   *
   * @param string $provider_id
   *   The provider ID.
   *
   * @return string|null
   *   The package name or NULL if not found.
   */
  public static function getPackage(string $provider_id): ?string {
    $provider = self::getProvider($provider_id);
    return $provider['package'] ?? NULL;
  }

  /**
   * Gets module name for a provider.
   *
   * @param string $provider_id
   *   The provider ID.
   *
   * @return string|null
   *   The module name or NULL if not found.
   */
  public static function getModule(string $provider_id): ?string {
    $provider = self::getProvider($provider_id);
    return $provider['module'] ?? NULL;
  }

  /**
   * Gets config name for a provider.
   *
   * @param string $provider_id
   *   The provider ID.
   *
   * @return string|null
   *   The config name or NULL if not found.
   */
  public static function getConfigName(string $provider_id): ?string {
    $provider = self::getProvider($provider_id);
    return $provider['config_name'] ?? NULL;
  }

  /**
   * Gets form class for a provider.
   *
   * @param string $provider_id
   *   The provider ID.
   *
   * @return string|null
   *   The form class or NULL if not found.
   */
  public static function getFormClass(string $provider_id): ?string {
    $provider = self::getProvider($provider_id);
    return $provider['form_class'] ?? NULL;
  }

  /**
   * Checks if a provider is valid.
   *
   * @param string $provider_id
   *   The provider ID.
   *
   * @return bool
   *   TRUE if the provider is valid, FALSE otherwise.
   */
  public static function isValidProvider(string $provider_id): bool {
    return isset(self::PROVIDERS[$provider_id]);
  }

  /**
   * Gets all provider IDs.
   *
   * @return array
   *   Array of provider IDs.
   */
  public static function getProviderIds(): array {
    return array_keys(self::PROVIDERS);
  }

}
