<?php

namespace Drupal\ai_experience_wizard\Service;

use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Batch operations for AI provider installation.
 */
class AiProviderBatchOperations {

  /**
   * Pulls the AI provider package from composer.
   *
   * @param string $ai_provider
   *   The AI provider name.
   * @param array $context
   *   The batch context array.
   */
  public static function pullAiProvider(string $ai_provider, array &$context) {
    try {
      $context['message'] = t('Checking AI provider package...');

      /** @var \Drupal\ai_experience_wizard\Service\AiProviderInstaller $installer */
      $installer = \Drupal::service('ai_experience_wizard.ai_provider_installer');

      // Check if the provider is already installed (both package and module)
      if ($installer->isProviderInstalled($ai_provider)) {
        $context['results']['package_pulled'] = TRUE;
        $context['message'] = t('AI provider package and module are already installed.');
        $context['finished'] = 1.0;
        return;
      }

      // Check if package is installed but module not available yet.
      if ($installer->isPackageInstalled($ai_provider)) {
        $context['results']['package_pulled'] = TRUE;
        $context['message'] = t('AI provider package is installed, but module not yet available.');
        $context['finished'] = 1.0;
        return;
      }

      // Only install if neither package nor module is available.
      $context['message'] = t('Installing AI provider package and dependencies...');
      $success = $installer->requireProviderOnly($ai_provider);

      if ($success) {
        $context['results']['package_pulled'] = TRUE;
        $context['message'] = t('AI provider package and dependencies installed successfully.');
      }
else {
        $context['results']['package_pulled'] = FALSE;
        $context['message'] = t('Failed to install AI provider package. Please check the logs for more details.');
        // Log the error for debugging.
        \Drupal::logger('ai_experience_wizard')->error('Failed to install AI provider package: @provider', ['@provider' => $ai_provider]);
}

      $context['finished'] = 1.0;
    }
catch (\Exception $e) {
      $context['results']['package_pulled'] = FALSE;
      $context['message'] = t('Error: @message', ['@message' => $e->getMessage()]);
      \Drupal::logger('ai_experience_wizard')->error('Batch operation failed: @error', ['@error' => $e->getMessage()]);
      $context['finished'] = 1.0;
}
  }

  /**
   * Installs the AI provider module.
   *
   * @param string $ai_provider
   *   The AI provider name.
   * @param array $context
   *   The batch context array.
   */
  public static function installAiProvider(string $ai_provider, array &$context) {
    try {
      $context['message'] = t('Installing AI provider module...');

      // Get module name from provider manager.
      $provider_manager = \Drupal::service('ai_experience_wizard.provider_manager');

      if (!$provider_manager->isValidProvider($ai_provider)) {
        $context['results']['module_installed'] = FALSE;
        $context['message'] = t('Unknown AI provider: @provider', ['@provider' => $ai_provider]);
        $context['finished'] = 1.0;
        return;
      }

      $module = $provider_manager->getModule($ai_provider);

      // Check if module is already enabled.
      if (\Drupal::moduleHandler()->moduleExists($module)) {
        $context['results']['module_installed'] = TRUE;
        $context['message'] = t('AI provider module is already enabled.');
        $context['finished'] = 1.0;
        return;
      }

      // Enable the module.
      $module_installer = \Drupal::service('module_installer');
      $success = $module_installer->install([$module]);

      if ($success) {
        $context['results']['module_installed'] = TRUE;
        $context['message'] = t('AI provider module enabled successfully.');
      }
else {
        $context['results']['module_installed'] = FALSE;
        $context['message'] = t('Failed to enable AI provider module.');
}

// 100% complete
      $context['finished'] = 1.0;
    }
catch (\Exception $e) {
      $context['results']['module_installed'] = FALSE;
      $context['message'] = t('Error: @message', ['@message' => $e->getMessage()]);
      \Drupal::logger('ai_experience_wizard')->error('Batch operation failed: @error', ['@error' => $e->getMessage()]);
      $context['finished'] = 1.0;
}
  }

  /**
   * Batch finished callback.
   *
   * @param bool $success
   *   Whether the batch completed successfully.
   * @param array $results
   *   The batch results.
   * @param array $operations
   *   The batch operations.
   */
  public static function batchFinished($success, $results, $operations) {
    if ($success) {
      $package_pulled = $results['package_pulled'] ?? FALSE;
      $module_installed = $results['module_installed'] ?? FALSE;

      // Get the AI provider from the first operation.
      $ai_provider = '';
      if (!empty($operations) && !empty($operations[0][1])) {
        $ai_provider = $operations[0][1][0] ?? '';
      }

      // Get module display name from provider manager.
      $provider_manager = \Drupal::service('ai_experience_wizard.provider_manager');
      $provider_info = $provider_manager->getProviderInfo($ai_provider);
      $module_display_name = $provider_info ? $provider_info['label'] : 'AI Provider';

      if ($package_pulled && $module_installed) {
        \Drupal::messenger()->addMessage(t('@module has been installed and enabled successfully.', ['@module' => $module_display_name]));

        // Redirect to the configuration form.
        $response = new RedirectResponse(
          Url::fromRoute('ai_experience_wizard.wizard', [], ['query' => ['step' => 2]])->toString()
        );
        $response->send();
        exit;
      }
elseif ($package_pulled) {
        \Drupal::messenger()->addMessage(t('@module package was installed successfully, but module enabling failed.', ['@module' => $module_display_name]));
}
else {
        \Drupal::messenger()->addError(t('There was an error during the @module installation process.', ['@module' => $module_display_name]));
}
    }
else {
      \Drupal::messenger()->addError(t('AI Provider installation failed. Please check the logs for more details.'));
}
  }

}
