<?php

declare(strict_types=1);

namespace Drupal\ai_experience_wizard\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\key\KeyRepositoryInterface;
use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;

/**
 * Service for managing Key entities for AI providers.
 */
final class KeyManager implements LoggerAwareInterface {

  use LoggerAwareTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * Constructs a new KeyManager object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, KeyRepositoryInterface $key_repository) {
    $this->entityTypeManager = $entity_type_manager;
    $this->keyRepository = $key_repository;
  }

  /**
   * Creates or retrieves a key for the given provider and field.
   *
   * @param string $provider
   *   The AI provider name (e.g., 'openai', 'anthropic').
   * @param string $field_name
   *   The field name (e.g., 'api_key', 'secret_key').
   * @param string $key_value
   *   The actual key value to store.
   *
   * @return string
   *   The key ID of the created or existing key.
   *
   * @throws \Exception
   */
  public function createOrGetKey(string $provider, string $field_name, string $key_value): string {
    $key_id = $this->generateKeyId($provider, $field_name);

    // Check if key already exists
    $existing_key = $this->keyRepository->getKey($key_id);
    if ($existing_key) {
      $this->logger->info('Reusing existing key: @key_id', ['@key_id' => $key_id]);
      return $key_id;
    }

    // Create new key
    $key_label = $this->generateKeyLabel($provider, $field_name);

    $values = [
      'label' => $key_label,
      'id' => $key_id,
      'key_type' => 'authentication',
      'key_type_settings' => [],
      'key_provider' => 'config',
      'key_provider_settings' => [
        'key_value' => $key_value,
      ],
      'key_input' => 'text_field',
      'description' => 'Auto-created by AI Experience Wizard',
    ];

    try {
      $key = $this->entityTypeManager->getStorage('key')->create($values);
      $key->save();

      $this->logger->info('Created new key: @key_id', ['@key_id' => $key_id]);
      return $key_id;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to create key @key_id: @error', [
        '@key_id' => $key_id,
        '@error' => $e->getMessage(),
      ]);
      throw new \Exception('Failed to create key: ' . $e->getMessage(), 0, $e);
    }
  }

  /**
   * Generates a standardized key ID.
   *
   * @param string $provider
   *   The AI provider name.
   * @param string $field_name
   *   The field name.
   *
   * @return string
   *   The generated key ID.
   */
  protected function generateKeyId(string $provider, string $field_name): string {
    return 'ai_experience_wizard_' . $provider . '_' . $field_name;
  }

  /**
   * Generates a human-readable key label.
   *
   * @param string $provider
   *   The AI provider name.
   * @param string $field_name
   *   The field name.
   *
   * @return string
   *   The generated key label.
   */
  protected function generateKeyLabel(string $provider, string $field_name): string {
    $provider_display = ucfirst($provider);
    $field_display = str_replace('_', ' ', ucwords($field_name, '_'));
    return "AI Experience Wizard - {$provider_display} {$field_display}";
  }

}
