<?php

namespace Drupal\ai_featured_image\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\file\Entity\File;
use Drupal\node\NodeInterface;
use GuzzleHttp\ClientInterface;
use Drupal\file\FileRepositoryInterface;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;

class FeaturedImageGenerator {

  protected ClientInterface $httpClient;
  protected FileRepositoryInterface $fileRepository;
  protected StreamWrapperManagerInterface $streamWrapperManager;
  protected ConfigFactoryInterface $configFactory;

  public function __construct(
    ClientInterface $httpClient,
    FileRepositoryInterface $fileRepository,
    StreamWrapperManagerInterface $streamWrapperManager,
    ConfigFactoryInterface $configFactory
  ) {
    $this->httpClient = $httpClient;
    $this->fileRepository = $fileRepository;
    $this->streamWrapperManager = $streamWrapperManager;
    $this->configFactory = $configFactory;
  }

  public function generateImageForNode(NodeInterface $node): ?File {
    $config = $this->configFactory->get('ai_featured_image.settings');
    $apiKey = $config->get('openai_api_key');
    $prompt_template = $config->get('prompt_template') ?: 'Header image about: [title]';

    $body = $node->hasField('body') && !$node->get('body')->isEmpty()
        ? $node->get('body')->value
        : '';
    $prompt = strtr($prompt_template, [
        '[title]' => $node->getTitle(),
        '[body]' => $body,
    ]);
    
    try {
      $response = $this->httpClient->post('https://api.openai.com/v1/images/generations', [
        'headers' => [
          'Authorization' => 'Bearer ' . $apiKey,
          'Content-Type' => 'application/json',
        ],
        'json' => [
          'prompt' => $prompt,
          'n' => 1,
          'size' => '1024x1024',
        ],
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);
      if (empty($data['data'][0]['url'])) {
        return NULL;
      }

      $image_url = $data['data'][0]['url'];
      $image_data = file_get_contents($image_url);
      $directory = 'public://ai_images';
      \Drupal::service('file_system')->prepareDirectory($directory, \Drupal\Core\File\FileSystemInterface::CREATE_DIRECTORY);

      $filename = 'featured-' . time() . '.png';
      $destination = $directory . '/' . $filename;
      \Drupal::service('file_system')->saveData($image_data, $destination, \Drupal\Core\File\FileSystemInterface::EXISTS_REPLACE);

      return $this->fileRepository->writeData($image_data, $destination, \Drupal\Core\File\FileSystemInterface::EXISTS_REPLACE);
    }
    catch (\Exception $e) {
      \Drupal::logger('ai_featured_image')->error('Error generating image: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }
}
