<?php

declare(strict_types=1);

namespace Drupal\ai_generation\Traits;
use Drupal\Core\Site\Settings;

trait ScaffoldGeneratorTrait {

  /**
   * Generate module/theme/config files based on AI JSON data.
   *
   * @param array $data
   *   The decoded JSON response from AI, must contain 'type', 'base_path', 'files'.
   *
   * @return array{written:int,skipped:int,paths:string[]}
   *   Stats and list of paths created.
   */
  protected function generateScaffold(array $data): array {
    $base = (string) ($data['base_path'] ?? '');
    $type = strtolower((string) ($data['type'] ?? ''));

    if ($base !== 'modules/custom' && $base !== 'themes/custom') {
      $base = ($type === 'theme') ? 'themes/custom' : 'modules/custom';
    }

    if($type === 'configuration'){
      $base = Settings::get('config_sync_directory');
    }

    $written = 0;
    $skipped = 0;

    foreach ($data['files'] as $file) {
      $rel = ltrim((string) ($file['path'] ?? ''), '/');
      $contents = (string) ($file['contents'] ?? '');

      if ($rel === '') {
        continue;
      }

      // AI already returns this info, just to be sure
      if (str_starts_with($rel, 'modules/') || str_starts_with($rel, 'themes/')) {
        $target = $rel;
      }
      else {
        $target = rtrim($base, '/') . '/' . $rel;
      }

      $dir = dirname($target);
      if (!is_dir($dir)) {
        mkdir($dir, 0775, true);
      }

      file_put_contents($target, $contents);
      $written++;
    }

    return [
      'written' => $written,
      'type' => $type,
      'skipped' => $skipped,
      'files' => implode("\n", array_column($data['files'], 'path')),
      'packageName' => $data['packageName']
    ];
  }

}
