<?php

namespace Drupal\ai_generation\Drush\Commands;

use Drupal\ai_generation\Service\Connection;
use Drush\Attributes as CLI;
use Drush\Commands\AutowireTrait;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Drupal\ai_generation\Constants\SuggestedSystemPrompt;
use Drupal\ai_generation\Traits\ScaffoldGeneratorTrait;

/**
 * A Drush commandfile.
 */
final class AiGenerationCommands extends DrushCommands {

  use AutowireTrait, ScaffoldGeneratorTrait;

  /**
   * Constructs an AiGenerationCommands object.
   */
  public function __construct(
    #[Autowire(service: 'ai_generation.connection')]
    private readonly Connection $providerManager,
  ) {
    parent::__construct();
  }

  /**
   * Command description here.
   */
  #[CLI\Command(name: 'ai_generation:generate', aliases: ['aigen'])]
  #[CLI\argument('prompt', 'Description of what you want to generate.')]
  #[CLI\Option('systemPrompt', 'Base output folder (default: modules/custom).')]
  // #[CLI\Option('dry-run', 'Do not write files, only display manifest.')]
  public function generate($prompt,
    array $options = [
      'systemPrompt' => SuggestedSystemPrompt::MIXED,
    ]
  ): void {

    $provider = $this->providerManager->getProvider();
    $model = $this->providerManager->getModel();
    $this->io()->title('AI CodeGen');
    $this->io()->writeln("Provider: <info>{$provider}</info>");
    $this->io()->writeln("Model   : <info>{$model}</info>");
    $this->io()->writeln("System prompt: <info>{$options['systemPrompt']}</info>");

    $chat = $this->providerManager->chat("Request: {$prompt}", $options['systemPrompt']);
    $data = json_decode((string) $chat, true);

    $result = $this->generateScaffold($data);

    $this->io()->text('The following files have been created.' . "\n" . $result['files']);
    $packageName = $result['packageName'] ?? null;
    
    if ($packageName) {
      $name = $result['type'];
      $enable = $this->io()->confirm("Would you like to enable the <info>{$packageName}</info> $name now?", false);
      if ($enable) {
        if($name == 'module'){
          exec("drush en -y " . escapeshellarg($packageName));
        }else{
          exec("drush theme:install -y " . escapeshellarg($packageName));
        }

        exec("drush cr");
        $this->io()->success("$name {$packageName} enabled and cleaned caches.");
      } else {
        $this->io()->note("You can enabled it later");
      }
    }
  }

}