<?php
namespace Drupal\ai_generation\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\ai_generation\Constants\SuggestedSystemPrompt;
use Drupal\ai_generation\Traits\ScaffoldGeneratorTrait;

class AiGenerationSettingsForm extends ConfigFormBase {

  use ScaffoldGeneratorTrait;

  /** @var \Drupal\ai_generation\Connection */
  protected $providerManager;

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ai_generation_settings_form';
  }


  public static function create(ContainerInterface $container): self {
    /** @var self $instance */
    $instance = parent::create($container);
    $instance->providerManager = $container->get('ai_generation.connection');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['ai_generation.settings'];
  }

  /**
   * {@inheritdoc}Gen
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    // $form['option'] = [
    //   '#type' => 'radios',
    //   '#title' => $this->t('What do you want to create?'),
    //   '#options' => [
    //     "modules/custom" => "Custom module",
    //     "config/sync" => "Yaml configuration"
    //   ],
    //     // '#description' => $this->t('Indicate the prompt you want to give to the AI. Be clear and concise.'),
    // ];

    $form['suggested'] = [
      '#type' => 'textarea',
      '#title' => $this->t('System prompt'),
      '#description' => $this->t('The prompt given to the AI to give the context.'),
      '#rows' => 15,
      '#attributes' => array('readonly' => 'readonly'),
      '#default_value' => SuggestedSystemPrompt::MIXED,
    ];

    $form['prompt'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Prompt'),
      '#description' => $this->t('Indicate the prompt you want to give to the AI. Be clear and concise.'),
    ];

    $form['ai_result'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'ai-result'],
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Generate'),
      '#ajax' => [
        'callback' => '::ajaxSubmit',
        'wrapper' => 'ai-result',
        'event' => 'click',
        'progress' => ['type' => 'throbber', 'message' => $this->t('Processing...')],
      ],
    ];

    return $form;
  }

  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $prompt = trim((string) $form_state->getValue('prompt'));
    if ($prompt === '') {
      $form_state->setErrorByName('prompt', $this->t('Please enter a prompt.'));
    }
  }

  public function ajaxSubmit(array &$form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();

    try {
      $prompt = (string) $form_state->getValue('prompt');
      $systemPrompt = (string) $form_state->getValue('suggested');
      $chat = $this->providerManager->chat("Request: $prompt", $systemPrompt);
      $data = json_decode((string) $chat, true);

      $result = [];
      switch($data['type']){
        case "module" || "theme"  || "configuration":
          $result = $this->generateScaffold($data);
          break;
        //---- TODO-----------
        // case "content":
        //   $result = $this->addContent($data);
        //   break;
        default:
          $result = [];
      }

      if(empty($result)){
        throw new \RuntimeException('No data found');
      }
      
      $html = '<pre>The following files have been created. <strong>Please review them before enabling it</strong>:' . "\n" . $result['files'] . '</pre>';

      $response->addCommand(new HtmlCommand('#ai-result', $html));
      $response->addCommand(new MessageCommand($this->t('Files written: @w', ['@w' => $result['written']])));
    } catch (\Throwable $e) {
      $response->addCommand(new MessageCommand($this->t('Error: @msg', ['@msg' => $e->getMessage()]), NULL, ['type' => 'error']));
    }

    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);
  }

}
