<?php

namespace Drupal\ai_generation\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\HtmlCommand;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\ai_generation\Constants\SuggestedSystemPrompt;
use Drupal\ai_generation\Traits\ScaffoldGeneratorTrait;

/**
 * Configure AI Generation settings for this site.
 */
class AiGenerationSettingsForm extends ConfigFormBase {

  use ScaffoldGeneratorTrait;

  /**
   * Provider manager service.
   *
   * @var \Drupal\ai_generation\Connection
   */
  protected $providerManager;

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ai_generation_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    /** @var self $instance */
    $instance = parent::create($container);
    $instance->providerManager = $container->get('ai_generation.connection');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['ai_generation.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    // --- Help and examples ------------------------------------------------
    $form['ai_warning'] = [
      '#type' => 'markup',
      '#markup' => '<div class="messages messages--warning">' .
      $this->t('⚠️ Please note: The AI is still learning and not everything it generates is guaranteed to be correct. Always review the generated code carefully before using it. 
      This is especially important for <strong>configuration</strong>, as invalid config files may cause errors during import.<br><br>
      The quality of the results will also depend on the <strong>model</strong> you are using — more powerful models usually provide more accurate and consistent outputs.') .
      '</div>',
    ];

    $form['help'] = [
      '#type' => 'details',
      '#title' => $this->t('What can I generate?'),
      '#open' => TRUE,
    ];

    $form['help']['intro'] = [
      '#markup' => $this->t('This assistant generates <strong>modules</strong>, <strong>recipes</strong>,<strong>themes</strong>, <strong>decoupled forms</strong>, and <strong>configuration</strong> for Drupal (YAML). <em>It does not create content</em> (nodes, users, taxonomy terms).'),
    ];

    $form['help']['examples'] = [
      '#theme' => 'item_list',
      '#items' => [
        $this->t('<strong>Configuration:</strong> Anything from a simple vocabulary "Tags" (no terms) to complex setups like a "Recipe" content type with fields, form/view displays, and a paginated view.'),
        $this->t('<strong>Module:</strong> From a minimal starter to a feature-rich custom module with services, event subscribers, permissions, routes, and controllers — according to your prompt.'),
        $this->t('<strong>Theme:</strong> From a starter theme to a fully configured theme with libraries, regions, breakpoints, settings, and Twig templates — as requested.'),
        $this->t('<strong>Recipe:</strong> A Drupal Recipe that bundles configuration and optional module dependencies.'),
        $this->t('<strong>Frontend (decoupled):</strong> A standalone Angular/React/Vue app placed under <code>frontend/{package}</code>, plus a small bridge module to expose it inside Drupal.'),
      ],
    ];

    $form['help_example'] = [
      '#type' => 'details',
      '#title' => $this->t('See sample prompts'),
      '#open' => FALSE,
    ];

    $form['help_example']['samples'] = [
      '#theme' => 'item_list',
      '#items' => [
        [
          '#markup' => '<strong>' . $this->t('Configuration:') . '</strong><br><pre>' .
          $this->t('Create a "Recipe" content type with an image field, a difficulty list (easy, medium, hard), and a time field in minutes. Add the default form and view displays, and also create a view at /recipes showing 12 items per page.') .
          '</pre>',
        ],
        [
          '#markup' => '<strong>' . $this->t('Module:') . '</strong><br><pre>' .
          $this->t('Create a module that allows me to register hotel reservations. Add attractive styles, make it accessible, and add unit tests.') .
          '</pre>',
        ],
        [
          '#markup' => '<strong>' . $this->t('Theme:') . '</strong><br><pre>' .
          $this->t('Create a custom theme named "my_theme" based on Olivero. Add two extra regions called "Header Top" and "Footer Secondary", and include a CSS and JS file through a library. Also define responsive breakpoints.') .
          '</pre>',
        ],
        [
          '#markup' => '<strong>' . $this->t('Recipe:') . '</strong><br><pre>' .
          $this->t('Create a recipe called "events_content" that provides an Event content type, installs the ds and fullcalendar_view modules, and grants permissions to content editors to manage events.') .
          '</pre>',
        ],
        [
          '#markup' => '<strong>' . $this->t('Frontend (decoupled):') . '</strong><br><pre>' .
          $this->t('Create a decoupled Angular app named "contact_form" with fields for Name, Email, and Message. Place it under frontend/contact_form, and generate a bridge module so that the app is accessible at /contact/form inside Drupal.') .
          '</pre>',
        ],
      ],
    ];

    // --- Configurable settings -------------------------------------------
    $form['timeout'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Timeout (seconds)'),
      '#default_value' => 300,
      '#description' => $this->t('Set the maximum wait time for the API call. The larger the request, the more time it will take and the more tokens will be consumed. <strong>Be careful.</strong>'),
    ];

    $form['prompt'] = [
      '#type' => 'textarea',
      '#required' => TRUE,
      '#required_error' => $this->t('Prompt is required.'),
      '#title' => $this->t('Prompt'),
      '#attributes' => ['class' => ['ai-prompt']],
      '#description' => $this->t('Specify what you need to generate. The functionality is for <strong>creation only</strong>, not editing, so the more specific and concise you are, the better results you will get.'),
    ];

    // Experimental voice to text in real time.Just works in English
    // $form['voice'] = [
    // '#type' => 'html_tag',
    // '#tag' => 'button',
    // '#value' => 'Hold to record 🎤',
    // '#attributes' => [
    // 'data-speech-record' => '1',
    // 'id' => 'ai-speech-btn',
    // 'title' => $this->t('Microphone'),
    // 'aria-label' => $this->t('Hold to talk'),
    // 'type' => 'button',
    // 'aria-pressed' => 'false',
    // 'class' => ['button', 'button--secondary'],
    // ],
    // ];.
    // $form['#attached']['library'][] = 'ai_generation/speech';.
    $form['ai_result'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'ai-result'],
    ];

    $ajaxMessage = $this->t('The more powerful the model, the longer it may take, so please be patient, this could takes a few minutes :)');
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Create'),
      '#ajax' => [
        'callback' => '::ajaxSubmit',
        'wrapper' => 'ai-result',
        'event' => 'click',
        'progress' => ['type' => 'throbber', 'message' => $ajaxMessage],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $prompt = trim((string) $form_state->getValue('prompt'));
    if ($prompt === '') {
      $form_state->setErrorByName('prompt', $this->t('Please enter a prompt.'));
    }
  }

  /**
   * AJAX form submission handler.
   */
  public function ajaxSubmit(array &$form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();

    try {
      $prompt = (string) $form_state->getValue('prompt');

      if (empty($prompt)) {
        $html = '<div class="messages messages--error">' . $this->t("Prompt is required") . '</div>';
      }
      else {
        $timeout = (int) $form_state->getValue("timeout");
        $ai_connection = $this->providerManager->chat($prompt, SuggestedSystemPrompt::systemText(), $timeout);

        if (empty($ai_connection)) {
          $html = '<div class="messages messages--error">' . $this->t("Received a not expected response. Try it again or maybe you can use another model") . '</div>';
        }
        else {
          $html = '<p><strong>Comments:</strong> ' . ($ai_connection['comments'] ?? 'N/A') . '</p>';

          if (!empty($result['files'])) {
            $html .= '<pre>The following files have been created. <strong>Please review them before enabling it</strong>:' . "\n" . $ai_connection['files'] . '</pre>';
            $response->addCommand(new MessageCommand($this->t('Files written: @w', ['@w' => $ai_connection['written']])));
          }
        }
      }
      $response->addCommand(new HtmlCommand('#ai-result', $html));
    }
    catch (\Throwable $e) {
      $response->addCommand(new MessageCommand($this->t('Error: @msg', ['@msg' => $e->getMessage()]), NULL, ['type' => 'error']));
    }

    return $response;
  }

}
