<?php

namespace Drupal\ai_generation\Drush\Commands;

use Drupal\ai_generation\Service\Connection;
use Drush\Attributes as CLI;
use Drush\Commands\AutowireTrait;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Drupal\ai_generation\Constants\SuggestedSystemPrompt;
use Drupal\ai_generation\Traits\ScaffoldGeneratorTrait;

/**
 * A Drush commandfile.
 */
final class AiGenerationCommands extends DrushCommands {

  use AutowireTrait, ScaffoldGeneratorTrait;

  /**
   * Constructs an AiGenerationCommands object.
   */
  public function __construct(
    #[Autowire(service: 'ai_generation.connection')]
    private readonly Connection $providerManager,
  ) {
    parent::__construct();
  }

  /**
   * Command description here.
   */
  #[CLI\Command(name: 'ai_generation:generate', aliases: ['aigen'])]
  #[CLI\argument('prompt', 'Description of what you want to generate.')]
  public function generate($prompt): int {

    if (empty($prompt)) {
      $this->io()->error("Prompt is required");
      return DrushCommands::EXIT_FAILURE;
    }
    $provider = $this->providerManager->getProvider();
    $model = $this->providerManager->getModel();
    $this->io()->title('AI CodeGen');
    $this->io()->writeln("Provider: <info>{$provider}</info>");
    $this->io()->writeln("Model   : <info>{$model}</info>");
    $this->io()->writeln("<info>The more powerful the model, the longer it may take, so please be patient, this could takes a few minutes :).</info>");

    $ai_connection = $this->providerManager->chat($prompt, SuggestedSystemPrompt::systemText());

    if (empty($ai_connection)) {
      $this->io()->error("Received a not expected response from $model. Try it again or maybe you can use another model.");
      return DrushCommands::EXIT_FAILURE;
    }

    if (!empty($ai_connection['files'])) {
      $this->io()->success("$provider response: " . ($ai_connection['comments'] ?? 'N/A'));

      $this->io()->text('The following files have been created:' . "\n" . $ai_connection['files']);
      $packageName = $ai_connection['packageName'] ?? NULL;

      if ($packageName) {
        $name = $ai_connection['type'];
        $enable = $this->io()->confirm("Would you like to enable the <info>{$packageName}</info> $name now?", FALSE);
        if ($enable) {
          if ($name == 'module') {
            exec("drush en -y " . escapeshellarg($packageName));
          }
          else {
            exec("drush theme:install -y " . escapeshellarg($packageName));
          }

          exec("drush cr");
          $this->io()->success("$name {$packageName} enabled and cleaned caches.");
        }
        else {
          $this->io()->note("You can enabled it later");
        }
      }
    }
    else {
      $this->io()->warning("$provider response: " . ($ai_connection['comments'] ?? 'N/A'));
    }

    return DrushCommands::EXIT_SUCCESS;
  }

}
