<?php

declare(strict_types=1);

namespace Drupal\ai_generation\Service;

use Psr\Log\LoggerInterface;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai_generation\Traits\ScaffoldGeneratorTrait;

/**
 * Small wrapper around drupal/ai to perform chat calls.
 */
final class Connection {

  use ScaffoldGeneratorTrait;

  /**
   * Provider api manager service.
   *
   * @var \Drupal\ai\ProviderManagerInterface */
  private $providerManager;

  /**
   * Provider logger service.
   *
   * @var \Psr\Log\LoggerInterface */
  private $logger;

  /**
   * The default provider/model set for "chat" operations.
   *
   * @var array
   *   The default provider/model set for "chat" operations.
   */
  private $sets;

  /**
   * Connection class constructor.
   *
   * @param mixed $provider_manager
   *   The 'ai.provider' plugin manager service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The 'logger.channel.ai_generation' logger service.
   */
  public function __construct($provider_manager, LoggerInterface $logger) {
    $this->providerManager = $provider_manager;
    $this->logger = $logger;
    $this->sets = $this->providerManager->getDefaultProviderForOperationType('chat');
  }

  /**
   * Executes a chat request and returns normalized text.
   *
   * @param string $userPrompt
   *   The user prompt.
   * @param string|null $systemPrompt
   *   System prompt. Defaults to a simple helper string.
   * @param int $timeout
   *   The timeout configured in AiGenerationSettingsForm.
   *
   * @return array
   *   The provider’s response.
   *
   * @throws \RuntimeException
   *   If the provider/model is not configured or the call fails.
   */
  public function chat(string $userPrompt, ?string $systemPrompt = 'You are a helpful Drupal assistant.', ?int $timeout = 300) : array {
    try {
      $configuration = [
        'http_client_options' => [
          'timeout' => $timeout,
        ],
      ];
      $provider = $this->providerManager->createInstance($this->getProvider(), $configuration);

      $messages = new ChatInput([
        new ChatMessage('system', $systemPrompt ?? ''),
        new ChatMessage('user', "Request: $userPrompt"),
      ]);

      $normalized = $provider->chat($messages, $this->getModel())->getNormalized();
      $aiResponse = (string) $normalized->getText();
      $data = json_decode((string) $aiResponse, TRUE) ?? [];
      return $this->generateScaffold($data);
    }
    catch (\Throwable $e) {
      $this->logger->error('AI chat failed: @msg', ['@msg' => $e->getMessage()]);
      throw new \RuntimeException('AI chat failed: ' . $e->getMessage(), 0, $e);
    }
  }

  /**
   * Returns the model either from drupal/ai defaults for "chat".
   *
   * @return string
   *   The ai's model.
   */
  public function getModel(): string {
    return $this->sets['model_id'] ?? throw new \RuntimeException('No default AI model configured for chatting.');
  }

  /**
   * Returns the provider either from drupal/ai defaults for "chat".
   *
   * @return string
   *   The ai's provider.
   */
  public function getProvider(): string {
    return $this->sets['provider_id'] ?? throw new \RuntimeException('No default AI provider configured for chatting.');
  }

}
