<?php

namespace Drupal\ai_generation\Drush\Commands;

use Drupal\ai_generation\Service\Connection;
use Drush\Attributes as CLI;
use Drush\Commands\AutowireTrait;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Drupal\ai_generation\Constants\SuggestedSystemPrompt;
use Drush\Boot\DrupalBootLevels;
use Drupal\ai_generation\Traits\ScaffoldGeneratorTrait;
use Drush\Attributes\ValidateModulesEnabled;

/**
 * AI generator drush command.
 *
 * @phpstan-consistent-constructor
 */
final class AiGenerationCommands extends DrushCommands {

  use AutowireTrait, ScaffoldGeneratorTrait;

  /**
   * Constructs an AiGenerationCommands object.
   */
  public function __construct(
    #[Autowire(service: 'ai_generation.connection')]
    private readonly Connection $providerManager,
  ) {
    parent::__construct();
  }

  /**
   * Command description here.
   */
  #[CLI\Command(name: 'ai_generation:generate', aliases: ['aigen'])]
  #[CLI\argument('prompt', 'Description of what you want to generate.')]
  #[CLI\Option('consumed-tokens', 'Get consumed used in the request')]
  #[CLI\Option('timeout', 'Set the maximum wait time for the API call. The larger the request, the more time it will take and the more tokens will be consumed. Be careful.')]
  #[CLI\Usage(name: 'drush aigen "Hello, generate a module that does ... features"', description: 'Interactive mode')]
  #[CLI\Usage(name: 'drush aigen "Hello, generate a theme that does ... features"', description: 'Interactive mode')]
  #[CLI\Usage(name: 'drush aigen "Hello, create a content type named event with ... fields"', description: 'Interactive mode')]
  #[CLI\Usage(name: 'drush aigen "Hello, create a view named event_results with ... features"', description: 'Interactive mode')]
  #[CLI\Bootstrap(level: DrupalBootLevels::FULL)]
  #[ValidateModulesEnabled(modules: ['ai', 'key'])]
  public function generate(
    string $prompt,
    array $options = [
      'timeout' => 300,
      'consumed-tokens' => TRUE,
    ],
  ): int {

    if (empty($prompt)) {
      $this->io()->error("Prompt is required");
      return DrushCommands::EXIT_FAILURE;
    }
    $provider = $this->providerManager->getProvider();
    $model = $this->providerManager->getModel();
    $this->io()->title('AI CodeGen');
    $this->io()->writeln("Provider: <info>{$provider}</info>");
    $this->io()->writeln("Model   : <info>{$model}</info>");
    $this->io()->writeln("<info>🤖 The more powerful the model, the longer it may take, so please be patient, this could takes a few minutes :).</info>");

    $ai_connection = $this->providerManager->chat($prompt, SuggestedSystemPrompt::gettext(0), $options['timeout']);

    if (empty($ai_connection)) {
      $this->io()->error("Received a not expected response from $model. Try it again or maybe you can use another model.");
      return DrushCommands::EXIT_FAILURE;
    }

    if ($options['consumed-tokens']) {
      $tokens = $ai_connection['tokens'];
      $this->io()->writeln("<comment>This request has consumed: $tokens tokens</comment>");
    }

    $time = $ai_connection['time'];
    if ($time) {
      $this->io()->writeln("<comment>This request has taken: $time seconds</comment>");
    }

    $this->formatAiComments($ai_connection['comments']);

    if (!empty($ai_connection['files'])) {
      $this->io()->text('The following files have been created:' . "\n" . $ai_connection['files']);
      $packageName = $ai_connection['packageName'] ?? NULL;

      if ($packageName) {
        $name = $ai_connection['type'];
        $enable = $this->io()->confirm("Would you like to enable the <info>{$packageName}</info> $name now?", FALSE);
        if ($enable) {
          if ($name == 'module') {
            exec("drush en -y " . escapeshellarg($packageName));
          }
          else {
            exec("drush theme:install -y " . escapeshellarg($packageName));
          }

          exec("drush cr");
          $this->io()->success("$name {$packageName} enabled and cleaned caches.");
        }
        else {
          $this->io()->note("You can enabled it later");
        }
      }
    }

    return DrushCommands::EXIT_SUCCESS;
  }

  /**
   * Format AI-generated comments into a string.
   *
   * @param string $comments
   *   The comments returned by the AI (string or array).
   *
   * @return string
   *   Formatted comments as a string.
   */
  private function formatAiComments($comments) {
    if (empty($comments)) {
      return '';
    }

    $this->output()->write("\r" . str_repeat(' ', 50) . "\r");
    $this->output()->writeln("<info>💬 AI Response:</info>");
    $this->output()->writeln("");

    $lines = explode("\n", $comments);
    foreach ($lines as $line) {
      $this->output()->writeln($line);
    }

    $this->output()->writeln("");
  }

}
