<?php

declare(strict_types=1);

namespace Drupal\ai_generation\Traits;

use Drupal\Core\Site\Settings;

/**
 * Declares a trait for create result content.
 */
trait ScaffoldGeneratorTrait {

  /**
   * Generate module/theme/config files based on AI JSON data.
   *
   * @param array $data
   *   The response from AI, must contain 'type', 'base_path', 'files'.
   * @param int $consumedTokens
   *   The total consumed tokens in the request.
   * @param string $time
   *   The total consumed tokens in the request.
   *
   * @return array{written:int,skipped:int,paths:string[]}
   *   Stats and list of paths created.
   */
  protected function generateScaffold(array $data, $consumedTokens, $time): array {

    if (empty($data)) {
      return [];
    }
    $type = strtolower((string) ($data['type'] ?? ''));
    $base = '';
    switch ($type) {
      case "module":
        $base = 'modules/custom';
        break;

      case "theme":
        $base = 'themes/custom';
        break;

      case "configuration":
        $base = Settings::get('config_sync_directory') ?: '../config/sync';
        break;

      case "recipe":
        $base = '../recipes';
        break;

      case "frontend":
        $base = '../apps';
        break;
    }

    $written = 0;
    $files = $data['files'];
    foreach ($files as $file) {
      $rel = ltrim((string) ($file['path'] ?? ''), '/');
      $contents = (string) ($file['contents'] ?? '');

      if ($rel === '') {
        continue;
      }

      $target = rtrim($base, '/') . '/' . $rel;
      $dir = dirname($target);
      if (!is_dir($dir)) {
        mkdir($dir, 0775, TRUE);
      }
      if (file_exists($target)) {
        continue;
      }
      file_put_contents($target, $contents);
      $written++;
    }

    return [
      'written' => $written,
      'type' => $type,
      'files' => implode("\n", array_column($data['files'], 'path')),
      'packageName' => $data['packageName'] ?? '',
      'comments' => $data['comments'],
      'tokens' => $consumedTokens,
      'time' => $time,
    ];
  }

}
