<?php

declare(strict_types=1);

namespace Drupal\ai_guardrails\Plugin\AiGuardrail;

use Drupal\ai\Guardrail\AiGuardrailPluginBase;
use Drupal\ai\Guardrail\NeedsAiPluginManagerInterface;
use Drupal\ai\Guardrail\NeedsAiPluginManagerTrait;
use Drupal\ai\Guardrail\Result\GuardrailResultInterface;
use Drupal\ai\Guardrail\Result\PassResult;
use Drupal\ai\Guardrail\Result\StopResult;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\ChatOutput;
use Drupal\Component\Plugin\ConfigurableInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\AiGuardrail;
use Drupal\Core\Utility\Token;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the Restrict to Topic guardrail.
 */
#[AiGuardrail(
  id: 'restrict_to_topic',
  label: new TranslatableMarkup('Restrict to Topic'),
  description: new TranslatableMarkup(
    "Checks if text's main topic is specified within a list of valid topics."
  ),
)]
final class RestrictToTopic extends AiGuardrailPluginBase implements ConfigurableInterface, PluginFormInterface, NeedsAiPluginManagerInterface, ContainerFactoryPluginInterface {

  use NeedsAiPluginManagerTrait;
  use StringTranslationTrait;

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  private Token $token;

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->setConfiguration($configuration);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    $instance = new RestrictToTopic(
      $configuration,
      $plugin_id,
      $plugin_definition,
    );

    $token = $container->get('token');
    if ($token instanceof Token) {
      $instance->token = $token;
    }

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration(): array {
    return $this->configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration): void {
    $this->configuration = $configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(
    array $form,
    FormStateInterface $form_state,
  ): array {
    $form['valid_topics'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Valid Topics'),
      '#description' => $this->t('List of valid topics, one per line.'),
      '#default_value' => $this->configuration['valid_topics'] ?? '',
    ];

    $form['invalid_topics'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Invalid Topics'),
      '#description' => $this->t('List of invalid topics, one per line.'),
      '#default_value' => $this->configuration['invalid_topics'] ?? '',
    ];

    $form['invalid_topics_present_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Message to send if invalid topics are present'),
      '#default_value' => $this->configuration['invalid_topics_present_message'] ?: 'The text contains invalid topics: @topics',
      '#description' => $this->t('You can use the placeholder %placeholder to include the list of invalid topics found.', [
        '%placeholder' => '@topics',
      ]),
    ];

    $form['valid_topics_missing_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Message to send if no valid topics are found'),
      '#default_value' => $this->configuration['valid_topics_missing_message'] ?: 'The text does not contain any of the valid topics: @topics',
      '#description' => $this->t('You can use the placeholder %placeholder to include the list of invalid topics found.', [
        '%placeholder' => '@topics',
      ]),
    ];

    $form['model'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Model'),
      '#description' => $this->t('The AI model to use for topic detection.'),
      '#default_value' => $this->configuration['model'] ?? 'openai/gpt-3.5-turbo',
    ];

    $form['model_configuration'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Model Configuration'),
      '#description' => $this->t('Configuration for the AI model, in JSON format.'),
      '#default_value' => $this->configuration['model_configuration'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(
    array &$form,
    FormStateInterface $form_state,
  ): void {}

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(
    array &$form,
    FormStateInterface $form_state,
  ): void {
    $values = $form_state->getValues();
    $this->setConfiguration($values);
  }

  /**
   * {@inheritdoc}
   */
  public function processInput(ChatInput $input): GuardrailResultInterface {
    $messages = $input->getMessages();
    $last_message = end($messages);

    if (!$last_message instanceof ChatMessage) {
      return new PassResult('No text message found to analyze.');
    }

    $text = $last_message->getText();
    $valid_topics = array_filter(array_map('trim', explode("\n", $this->configuration['valid_topics'] ?? '')));
    $invalid_topics = array_filter(array_map('trim', explode("\n", $this->configuration['invalid_topics'] ?? '')));
    $all_topics = array_merge($valid_topics, $invalid_topics);
    $all_topics_formatted = implode(',', $all_topics);

    $prompt = <<<PROMPT
Given a text and a list of topics, return a valid json list of which topics are present in the text. If none, just return an empty list. Don't format the output in any other way, just return the json list.

Output Format:
-------------
"topics_present": []

Text:
----
"$text"

Topics:
------
$all_topics_formatted

Result:
------
PROMPT;

    $input = new ChatInput([
      new ChatMessage("user", $prompt),
    ]);

    $model_to_use = explode('/', $this->configuration['model']);
    $provider = $model_to_use[0] ?? 'openai';
    $model = $model_to_use[1] ?? 'gpt-3.5-turbo';

    $ai_provider = $this->getAiPluginManager()->createInstance($provider);

    $model_configuration = \json_decode($this->configuration['model_configuration'] ?? '', TRUE) ?: [];
    // @phpstan-ignore-next-line
    $ai_provider->setConfiguration($model_configuration);

    // @phpstan-ignore-next-line
    $response = $ai_provider
      ->chat($input, $model, ['ai'])
      ->getNormalized();
    $response_decoded = json_decode($response->getText());
    $topics_present = $response_decoded->topics_present ?? [];

    $invalid_topics_found = [];
    $valid_topics_found = [];
    foreach ($topics_present as $topic) {
      if (\in_array($topic, $valid_topics)) {
        $valid_topics_found[] = $topic;
      }
      elseif (\in_array($topic, $invalid_topics)) {
        $invalid_topics_found[] = $topic;
      }
    }

    if (\count($invalid_topics_found) > 0) {
      return new StopResult(
        $this->token->replace($this->configuration['invalid_topics_present_message'], [
          'custom' => ['@topics' => implode(', ', $invalid_topics_found)],
        ])
      );
    }

    if (\count($valid_topics) > 0 && \count($valid_topics_found) === 0) {
      return new StopResult(
        $this->token->replace($this->configuration['valid_topics_missing_message'], [
          'custom' => ['@topics' => implode(', ', $invalid_topics_found)],
        ])
      );
    }

    return new PassResult(
      $this->token->replace('The text contains valid topics: @topics', [
        'custom' => ['@topics' => implode(', ', $invalid_topics_found)],
      ])
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processOutput(ChatOutput $output): GuardrailResultInterface {
    // This guardrail only processes input, not output.
    return new PassResult('Output processing is not applicable for this guardrail.');
  }

}
