<?php

namespace Drupal\ai_integration_eca_agents\Plugin\AiAgent;

use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai_integration_eca_agents\EntityViolationException;
use Drupal\ai_integration_eca_agents\MissingEventException;
use Drupal\ai_integration_eca_agents\Schema\Eca as EcaSchema;
use Drupal\ai_integration_eca_agents\TypedData\EcaModelDefinition;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai_agents\Attribute\AiAgent;
use Drupal\ai_agents\PluginBase\AiAgentBase;
use Drupal\ai_agents\PluginInterfaces\AiAgentInterface;
use Drupal\ai_agents\Task\TaskInterface;
use Drupal\ai_integration_eca_agents\Services\DataProvider\DataProviderInterface;
use Drupal\ai_integration_eca_agents\Services\DataProvider\DataViewModeEnum;
use Drupal\ai_integration_eca_agents\Services\EcaRepository\EcaRepositoryInterface;
use Drupal\Core\TypedData\Exception\MissingDataException;
use Drupal\eca\Entity\Eca as EcaEntity;
use Illuminate\Support\Arr;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Serializer\Exception\ExceptionInterface;
use Symfony\Component\Serializer\SerializerInterface;

/**
 * The ECA agent.
 */
#[AiAgent(
  id: 'eca',
  label: new TranslatableMarkup('ECA Agent'),
)]
class Eca extends AiAgentBase implements ContainerFactoryPluginInterface {

  /**
   * The DTO.
   *
   * @var array
   */
  protected array $dto;

  /**
   * The ECA entity.
   *
   * @var \Drupal\eca\Entity\Eca|null
   */
  protected ?EcaEntity $model = NULL;

  /**
   * The ECA data provider.
   *
   * @var \Drupal\ai_integration_eca_agents\Services\DataProvider\DataProviderInterface
   */
  protected DataProviderInterface $dataProvider;

  /**
   * The ECA helper.
   *
   * @var \Drupal\ai_integration_eca_agents\Services\EcaRepository\EcaRepositoryInterface
   */
  protected EcaRepositoryInterface $ecaRepository;

  /**
   * The serializer.
   *
   * @var \Symfony\Component\Serializer\SerializerInterface
   */
  protected SerializerInterface $serializer;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->dataProvider = $container->get('ai_integration_eca_agents.services.data_provider');
    $instance->ecaRepository = $container->get('ai_integration_eca_agents.services.eca_repository');
    $instance->serializer = $container->get('serializer');
    $instance->dto = [
      'task_description' => '',
      'feedback' => '',
      'questions' => [],
      'data' => [],
      'logs' => [],
    ];

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getId(): string {
    return 'eca';
  }

  /**
   * {@inheritdoc}
   */
  public function agentsNames(): array {
    return ['Event-Condition-Action agent'];
  }

  /**
   * {@inheritdoc}
   */
  public function isAvailable(): bool {
    return $this->agentHelper->isModuleEnabled('eca');
  }

  /**
   * {@inheritdoc}
   */
  public function hasAccess() {
    if ($this->hasPermission()) {
      return parent::hasAccess();
    }
    return AccessResult::forbidden();
  }

  /**
   * {@inheritdoc}
   */
  public function agentsCapabilities(): array {
    return [
      'eca' => [
        'name' => 'Event-Condition-Action Agent',
        'description' => 'This is agent is capable of adding, editing or informing about Event-Condition-Action models on a Drupal website. Note that it does not add events, conditions or actions as those require a specific implementation via code.',
        'inputs' => [
          'free_text' => [
            'name' => 'Prompt',
            'type' => 'string',
            'description' => 'The prompt to create, edit or ask questions about Event-Condition-Actions models.',
            'default_value' => '',
          ],
        ],
        'outputs' => [
          'answers' => [
            'description' => 'The answers to the questions asked about the Event-Condition-Action model.',
            'type' => 'string',
          ],
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function determineSolvability(): int {
    parent::determineSolvability();
    try {
      $type = $this->determineTypeOfTask();
    }
    catch (\Exception) {
      $type = '';
    }
    return match ($type) {
      'create', 'edit' => AiAgentInterface::JOB_SOLVABLE,
      'info' => AiAgentInterface::JOB_SHOULD_ANSWER_QUESTION,
      'fail' => AiAgentInterface::JOB_NEEDS_ANSWERS,
      default => AiAgentInterface::JOB_NOT_SOLVABLE,
    };
  }

  /**
   * {@inheritdoc}
   */
  public function solve(): array|string {
    if (isset($this->dto['setup_agent']) && $this->dto['setup_agent'] === TRUE) {
      parent::determineSolvability();
    }

    switch (Arr::get($this->dto, 'data.0.action')) {
      case 'create':
      case 'edit':
        $this->buildModel();
        break;

      case 'info':
        $log = $this->answerQuestion();
        $this->dto['logs'][] = $log;
        break;
    }

    return Arr::join($this->dto['logs'], "\n");
  }

  /**
   * {@inheritdoc}
   */
  public function answerQuestion(): string|TranslatableMarkup {
    if (!$this->hasPermission()) {
      return $this->t('You do not have permission to do this.');
    }

    $this->dataProvider->setViewMode(DataViewModeEnum::Full);

    $context = [];
    if (!empty($this->model)) {
      $context['The information of the model, in JSON format'] = sprintf("```json\n%s\n```", Json::encode($this->dataProvider->getModels([$this->model->id()])));
    }
    elseif (!empty($this->data[0]['component_ids'])) {
      $context['The details about the components'] = sprintf("```json\n%s\n```", Json::encode($this->dataProvider->getComponents($this->data[0]['component_ids'])));
    }

    if (empty($context)) {
      return $this->t('Sorry, I could not answer your question without anymore context.');
    }

    // Perform the prompt.
    try {
      $data = $this->agentHelper->runSubAgent('answerQuestion', $context);
    }
    catch (\Exception $e) {
      return $this->t('Sorry, I could not answer your question without anymore context: @message', [
        '@message' => $e->getMessage(),
      ]);
    }
    if ($data instanceof ChatMessage) {
      return $data->getText();
    }
    if (!is_array($data) || empty($data[0]['action'])) {
      // Quit early if the returned response isn't what we expected.
      return 'Sorry, we could not understand what you wanted to do, please try again.';
    }

    if (isset($data[0]['answer'])) {
      $answer = array_map(function ($dataPoint) {
        return $dataPoint['answer'];
      }, $data);

      return implode("\n", $answer);
    }

    return $this->t('Sorry, I got no answers for you.');
  }

  /**
   * {@inheritdoc}
   */
  public function askQuestion(): array {
    return (array) Arr::get($this->dto, 'questions');
  }

  /**
   * {@inheritdoc}
   */
  public function approveSolution(): void {
    $this->dto = Arr::set($this->dto, 'data.0.action', 'create');
  }

  /**
   * {@inheritdoc}
   */
  public function setTask(TaskInterface $task): void {
    parent::setTask($task);

    $this->dto['task_description'] = $task->getDescription();
  }

  /**
   * Get the data transfer object.
   *
   * @return array
   *   Returns the data transfer object.
   */
  public function getDto(): array {
    return $this->dto;
  }

  /**
   * Set the data transfer object.
   *
   * @param array $dto
   *   The data transfer object.
   */
  public function setDto(array $dto): void {
    $this->dto = $dto;

    if (!empty($this->dto['model_id'])) {
      $this->model = $this->ecaRepository->get($this->dto['model_id']);
    }
  }

  /**
   * Determines if the current user has permission to use the agent.
   *
   * @return bool
   *   TRUE if the user has permission, FALSE otherwise.
   */
  protected function hasPermission(): bool {
    return $this->currentUser->hasPermission('administer eca')
      || $this->currentUser->hasPermission('modeler api administer eca')
      || $this->currentUser->hasPermission('modeler api edit eca');
  }

  /**
   * Determine the type of task.
   *
   * @return string
   *   Returns the type of task.
   *
   * @throws \Exception
   */
  protected function determineTypeOfTask(): string {
    $context = $this->getFullContextOfTask($this->task);
    $userContext = [
      'A summary of the existing models' => sprintf("```json\n%s\n```", Json::encode($this->dataProvider->getModels())),
      'The list of available events, conditions and actions, in JSON format' => sprintf("```json\n%s\n```", Json::encode($this->dataProvider->getComponents())),
    ];
    if (!empty($this->model)) {
      $context .= 'A model already exists, so creation is not possible.';
      $userContext['A summary of the existing models'] = sprintf("```json\n%s\n```", Json::encode($this->dataProvider->getModels([$this->model->id()])));
    }
    $userContext['Task description and if available comments description'] = $context;

    // Prepare and run the prompt by fetching all the relevant info.
    $data = $this->agentHelper->runSubAgent('determineTask', $userContext);

    if ($data instanceof ChatMessage) {
      $this->dto['questions'][] = $data->getText();
      return 'fail';
    }
    if (!is_array($data) || empty($data[0]['action'])) {
      // Quit early if the returned response isn't what we expected.
      $this->dto['questions'][] = 'Sorry, we could not understand what you wanted to do, please try again.';
      return 'fail';
    }

    if (in_array($data[0]['action'], [
      'create',
      'edit',
      'info',
    ])) {
      if (!empty($data[0]['model_id'])) {
        $this->dto['model_id'] = $data[0]['model_id'];
        $this->model = $this->ecaRepository->get($data[0]['model_id']);
      }

      if (!empty($data[0]['feedback'])) {
        $this->dto['feedback'] = $data[0]['feedback'];
      }

      $this->dto['data'] = $data;

      return $data[0]['action'];
    }

    throw new \Exception('Invalid action determined for ECA');
  }

  /**
   * Create a configuration item for ECA.
   */
  protected function buildModel(): void {
    $this->dataProvider->setViewMode(DataViewModeEnum::Full);

    // Prepare the prompt.
    $context = [];

    // The schema of the ECA-config that the LLM should follow.
    $definition = EcaModelDefinition::create();
    $schema = new EcaSchema($definition, $definition->getPropertyDefinitions());
    try {
      $schema = $this->serializer->serialize($schema, 'schema_json:json', []);
    }
    catch (ExceptionInterface $e) {
      $this->dto['logs'][] = 'Can not serialize the JSON Schema of the model: ' . $e->getMessage();
      return;
    }
    $context['JSON Schema of the model'] = sprintf("```json\n%s\n```", $schema);

    // The model that should be edited.
    if (!empty($this->model)) {
      $models = $this->dataProvider->getModels([$this->model->id()]);
      $context['The model to edit'] = sprintf("```json\n%s\n```", Json::encode(reset($models)));
    }

    // Components or plugins that the LLM should use.
    if (Arr::has($this->dto, 'data.0.component_ids')) {
      $componentIds = Arr::get($this->dto, 'data.0.component_ids', []);
      $context['The details about the components'] = sprintf("```json\n%s\n```", Json::encode($this->dataProvider->getComponents($componentIds)));
    }

    // Optional feedback that the previous prompt provided.
    if (Arr::has($this->dto, 'data.0.feedback')) {
      $context['Guidelines'] = Arr::get($this->dto, 'data.0.feedback');
    }

    // Execute it.
    try {
      $data = $this->agentHelper->runSubAgent('buildModel', $context);
    }
    catch (\Exception $e) {
      $this->dto['logs'][] = 'Problem when building the model with the LLM: ' . $e->getMessage();
      return;
    }
    if ($data instanceof ChatMessage) {
      $this->dto['logs'][] = 'Problem when building the model with the LLM: ' . $data->getText();
      return;
    }
    if (!is_array($data) || empty($data[0]['action'])) {
      // Quit early if the returned response isn't what we expected.
      $this->dto['logs'][] = 'Sorry, we could not understand what you wanted to do, please try again.';
      return;
    }

    $message = Arr::get($data, '0.message', 'Could not create ECA config.');
    if (
      Arr::get($data, '0.action', 'fail') !== 'build'
      || !Arr::has($data, '0.model')
    ) {
      $this->dto['logs'][] = 'Problem when building the model with the LLM: ' . $message;
      return;
    }

    $modelData = Arr::get($data, '0.model');
    if (is_string($modelData)) {
      try {
        $modelData = json_decode($modelData, TRUE, 512, JSON_THROW_ON_ERROR);
      }
      catch (\JsonException $e) {
        $this->dto['logs'][] = $this->t('The created model data can not be parsed: @message', ['@message' => $e->getMessage()]);
      }
    }
    if (is_array($modelData)) {
      try {
        $eca = $this->ecaRepository->build($modelData, TRUE, $this->model?->id());
        $message = $this->model ? $this->t('Model "@model" was altered.', ['@model' => $this->model->label()]) : $this->t('A new model was created: "@model".', ['@model' => $eca->label()]);
      }
      catch (MissingDataException | EntityViolationException | MissingEventException $e) {
        $message = $this->t('Error when creating ECA model: @message', [
          '@message' => $e->getMessage(),
        ]);
      }
    }
    else {
      $message = $this->t('Unable create ECA model.');
    }

    $this->dto['logs'][] = $message;
  }

}
