<?php

namespace Drupal\ai_llms_txt_generator\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\ai_llms_txt_generator\Service\SitemapParserService;
use Drupal\ai_llms_txt_generator\Service\AiGeneratorService;

/**
 * Configuration form for AI LLMs.txt Generator settings.
 */
class LlmsTxtSettingsForm extends ConfigFormBase {

  /**
   * The sitemap parser service.
   *
   * @var \Drupal\ai_llms_txt_generator\Service\SitemapParserService
   */
  protected $sitemapParser;

  /**
   * The AI generator service.
   *
   * @var \Drupal\ai_llms_txt_generator\Service\AiGeneratorService
   */
  protected $aiGenerator;
  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * Constructs a LlmsTxtSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\ai_llms_txt_generator\Service\SitemapParserService $sitemap_parser
   *   The sitemap parser service.
   * @param \Drupal\ai_llms_txt_generator\Service\AiGeneratorService $ai_generator
   *   The AI generator service.\.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    SitemapParserService $sitemap_parser,
    AiGeneratorService $ai_generator,
    DateFormatterInterface $date_formatter,
  ) {
    parent::__construct($config_factory);
    $this->sitemapParser = $sitemap_parser;
    $this->aiGenerator = $ai_generator;
    $this->dateFormatter = $date_formatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('ai_llms_txt_generator.sitemap_parser'),
      $container->get('ai_llms_txt_generator.ai_generator'),
       $container->get('date.formatter')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['ai_llms_txt_generator.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_llms_txt_generator_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('ai_llms_txt_generator.settings');

    // Info message about AI provider.
    $form['ai_info'] = [
      '#type' => 'item',
      '#markup' => $this->t('This module uses the default AI provider configured in <a href="@url">AI module settings</a>. Make sure you have configured an AI provider with chat support before generating content.', [
        '@url' => '/admin/config/ai',
      ]),
    ];

    $form['site_info'] = [
      '#type' => 'details',
      '#title' => $this->t('Site Information'),
      '#open' => TRUE,
    ];

    $form['site_info']['site_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Site Name'),
      '#default_value' => $config->get('site_name') ?: $this->config('system.site')->get('name'),
      '#description' => $this->t('The name of your site for LLMs.txt.'),
    ];

    $form['site_info']['site_description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Site Description'),
      '#default_value' => $config->get('site_description') ?: '',
      '#description' => $this->t('A brief description of your site.'),
      '#rows' => 3,
    ];

    $form['sitemap'] = [
      '#type' => 'details',
      '#title' => $this->t('Sitemap Configuration'),
      '#open' => TRUE,
    ];

    $form['sitemap']['sitemap_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Sitemap URL'),
      '#default_value' => $config->get('sitemap_url') ?: '',
      '#description' => $this->t('Full URL to your sitemap.xml file (e.g., https://example.com/sitemap.xml)'),
      '#required' => TRUE,
    ];

    $form['generation'] = [
      '#type' => 'details',
      '#title' => $this->t('Content Generation'),
      '#open' => TRUE,
    ];

    $last_generated = $config->get('last_generated');
    if ($last_generated) {
      $form['generation']['last_generated'] = [
        '#type' => 'item',
        '#title' => $this->t('Last Generated'),
        '#markup' => $this->t('@time ago', [
          '@time' => $this->dateFormatter->formatInterval(time() - $last_generated),
        ]),
      ];
    }

    $form['generation']['generate_button'] = [
      '#type' => 'button',
      '#value' => $this->t('Generate LLMs.txt with AI'),
      '#ajax' => [
        'callback' => '::generateCallback',
        'wrapper' => 'llms-txt-generation-wrapper',
        'effect' => 'fade',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Generating content with AI...'),
        ],
      ],
    ];

    $form['generation']['wrapper_start'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'llms-txt-generation-wrapper'],
    ];

    $form['generation']['wrapper_start']['messages'] = [
      '#type' => 'status_messages',
    ];

    $form['generation']['wrapper_start']['generated_content'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Generated LLMs.txt Content'),
      '#default_value' => $config->get('generated_content') ?: '',
      '#description' => $this->t('AI-generated LLMs.txt content. You can edit this manually if needed.'),
      '#rows' => 20,
    ];

    $form['manual_content'] = [
      '#type' => 'details',
      '#title' => $this->t('Manual Content (Optional)'),
      '#open' => FALSE,
    ];

    $form['manual_content']['content'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Additional Manual Content'),
      '#default_value' => $config->get('content') ?: '',
      '#description' => $this->t('Optional additional content to append to the generated LLMs.txt.'),
      '#rows' => 10,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * AJAX callback for generate button.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form element to return.
   */
  public function generateCallback(array &$form, FormStateInterface $form_state) {
    $config = $this->config('ai_llms_txt_generator.settings');
    $sitemap_url = $config->get('sitemap_url');

    if (empty($sitemap_url)) {
      $this->messenger()->addError($this->t('Please configure a sitemap URL first.'));
      return $form['generation']['wrapper_start'];
    }

    // Parse sitemap.
    $urls = $this->sitemapParser->parseSitemap($sitemap_url);

    if (empty($urls)) {
      $this->messenger()->addError($this->t('Failed to parse sitemap or no URLs found.'));
      return $form['generation']['wrapper_start'];
    }

    // Format for AI.
    $sitemap_data = $this->sitemapParser->formatForAi($urls);

    // Generate content with AI.
    $generated_content = $this->aiGenerator->generateContent($sitemap_data);

    if ($generated_content === NULL) {
      $this->messenger()->addError($this->t('Failed to generate content with AI. Please check your AI provider configuration and logs.'));
      return $form['generation']['wrapper_start'];
    }

    // Update the textarea with generated content.
    $form['generation']['wrapper_start']['generated_content']['#value'] = $generated_content;

    // Save to config.
    $this->configFactory()->getEditable('ai_llms_txt_generator.settings')
      ->set('generated_content', $generated_content)
      ->set('last_generated', time())
      ->save();

    $this->messenger()->addStatus($this->t('LLMs.txt content generated successfully! @words words generated.', [
      '@words' => str_word_count($generated_content),
    ]));

    return $form['generation']['wrapper_start'];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('ai_llms_txt_generator.settings')
      ->set('site_name', $form_state->getValue('site_name'))
      ->set('site_description', $form_state->getValue('site_description'))
      ->set('sitemap_url', $form_state->getValue('sitemap_url'))
      ->set('generated_content', $form_state->getValue('generated_content'))
      ->set('content', $form_state->getValue('content'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
