<?php

namespace Drupal\ai_llms_txt_generator\Service;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service to generate LLMs.txt content using AI.
 */
class AiGeneratorService {

  /**
   * The AI provider manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $aiProvider;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructs an AiGeneratorService object.
   *
   * @param \Drupal\ai\AiProviderPluginManager $ai_provider
   *   The AI provider plugin manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   */
  public function __construct(
    AiProviderPluginManager $ai_provider,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
  ) {
    $this->aiProvider = $ai_provider;
    $this->configFactory = $config_factory;
    $this->logger = $logger_factory->get('ai_llms_txt_generator');
  }

  /**
   * Generate LLMs.txt content using AI.
   *
   * @param string $sitemap_data
   *   Formatted sitemap data.
   * @param array $options
   *   Additional options for generation.
   *
   * @return string|null
   *   Generated LLMs.txt content or NULL on failure.
   */
  public function generateContent(string $sitemap_data, array $options = []): ?string {
    try {
      // Get default provider settings for chat operation.
      $aiProviderData = $this->aiProvider->getDefaultProviderForOperationType('chat');

      if (empty($aiProviderData)) {
        $this->logger->error('No default AI provider configured for chat operation');
        return NULL;
      }

      // Create provider instance.
      $provider = $this->aiProvider->createInstance($aiProviderData['provider_id']);

      // Build the prompt.
      $prompt = $this->buildPrompt($sitemap_data, $options);

      // Create chat input with system and user messages.
      $messages = new ChatInput([
        new ChatMessage(
          'system',
          'You are an expert in creating LLMs.txt files following the specification at https://llmstxt.org/. Generate well-structured, compliant LLMs.txt content.'
        ),
        new ChatMessage('user', $prompt),
      ]);

      // Make the API call.
      $response = $provider->chat($messages, $aiProviderData['model_id'])->getNormalized();

      if (empty($response)) {
        $this->logger->error('AI provider returned empty content');
        return NULL;
      }

      // Extract text content from ChatMessage object.
      $generated_text = '';
      if ($response instanceof ChatMessage) {
        $generated_text = $response->getText();
      }
      elseif (is_string($response)) {
        $generated_text = $response;
      }
      else {
        $this->logger->error('Unexpected response type from AI provider');
        return NULL;
      }

      if (empty($generated_text)) {
        $this->logger->error('AI provider returned empty text content');
        return NULL;
      }

      // Clean and return the generated content.
      return $this->cleanGeneratedContent($generated_text);
    }
    catch (\Exception $e) {
      $this->logger->error('Error generating content with AI: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Build the prompt for AI generation.
   *
   * @param string $sitemap_data
   *   Formatted sitemap data.
   * @param array $options
   *   Additional options.
   *
   * @return string
   *   The prompt for AI.
   */
  protected function buildPrompt(string $sitemap_data, array $options): string {
    $config = $this->configFactory->get('ai_llms_txt_generator.settings');
    $site_name = $config->get('site_name') ?: $this->configFactory->get('system.site')->get('name');
    $site_description = $config->get('site_description') ?: '';

    $prompt = "Create an LLMs.txt file following the specification at https://llmstxt.org/.\n\n";
    $prompt .= "Site Information:\n";
    $prompt .= "- Name: {$site_name}\n";

    if ($site_description) {
      $prompt .= "- Description: {$site_description}\n";
    }

    $prompt .= "\n{$sitemap_data}\n\n";

    $prompt .= "IMPORTANT REQUIREMENTS:\n\n";

    $prompt .= "1. FORMAT:\n";
    $prompt .= "   - Use markdown formatting with clear hierarchy (# ## ###)\n";
    $prompt .= "   - Start with site name as H1 and one-sentence description in blockquote\n";
    $prompt .= "   - Keep it concise and scannable for AI systems\n\n";

    $prompt .= "2. INCLUDE ACTUAL URLS:\n";
    $prompt .= "   - List the 10-15 most important pages with their actual URLs from the sitemap\n";
    $prompt .= "   - Format as: [Page Title](URL) - Brief description\n";
    $prompt .= "   - Group URLs by category/topic\n";
    $prompt .= "   - Prioritize high-priority URLs from the sitemap\n\n";

    $prompt .= "3. STRUCTURE:\n";
    $prompt .= "   - Overview: 2-3 sentences about the site's purpose\n";
    $prompt .= "   - Key Pages: List important pages with URLs and descriptions\n";
    $prompt .= "   - Main Topics: Categorize content by theme\n";
    $prompt .= "   - Site Structure: Show major sections/paths\n\n";

    $prompt .= "4. STYLE:\n";
    $prompt .= "   - Be concise - aim for 200-400 words total\n";
    $prompt .= "   - Write for AI consumption, not humans\n";
    $prompt .= "   - Focus on facts and structure, not marketing language\n";
    $prompt .= "   - Use bullet points and lists for scannability\n\n";

    $prompt .= "5. EXAMPLE FORMAT:\n";
    $prompt .= "```\n";
    $prompt .= "# Site Name\n\n";
    $prompt .= "> One-sentence description of what this site does\n\n";
    $prompt .= "## Overview\n";
    $prompt .= "Brief 2-3 sentence overview of the site's purpose and audience.\n\n";
    $prompt .= "## Key Pages\n";
    $prompt .= "- [About](/about) - Company information and mission\n";
    $prompt .= "- [Products](/products) - Product catalog and features\n";
    $prompt .= "- [Documentation](/docs) - Technical documentation and guides\n\n";
    $prompt .= "## Main Topics\n";
    $prompt .= "### Category 1\n";
    $prompt .= "Brief description of this content area\n\n";
    $prompt .= "### Category 2\n";
    $prompt .= "Brief description of this content area\n\n";
    $prompt .= "## Site Structure\n";
    $prompt .= "/section/ - Description of section\n";
    $prompt .= "/another/ - Description of another section\n";
    $prompt .= "```\n\n";

    $prompt .= "Generate ONLY the LLMs.txt content following this format. No explanations, no preamble, no code blocks around the output.";

    return $prompt;
  }

  /**
   * Clean and format the generated content.
   *
   * @param string $content
   *   Raw generated content.
   *
   * @return string
   *   Cleaned content.
   */
  protected function cleanGeneratedContent(string $content): string {
    // Remove markdown code blocks if present.
    $content = preg_replace('/^```[a-z]*\n/m', '', $content);
    $content = preg_replace('/\n```$/m', '', $content);

    // Trim whitespace.
    $content = trim($content);

    return $content;
  }

}
