<?php

namespace Drupal\ai_llms_txt_generator\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\ai_llms_txt_generator\Service\SitemapParserService;
use Drupal\ai_llms_txt_generator\Service\AiGeneratorService;

/**
 * Controller for generating LLMs.txt content.
 */
class GenerateController extends ControllerBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The sitemap parser service.
   *
   * @var \Drupal\ai_llms_txt_generator\Service\SitemapParserService
   */
  protected $sitemapParser;

  /**
   * The AI generator service.
   *
   * @var \Drupal\ai_llms_txt_generator\Service\AiGeneratorService
   */
  protected $aiGenerator;

  /**
   * Constructs a GenerateController object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\ai_llms_txt_generator\Service\SitemapParserService $sitemap_parser
   *   The sitemap parser service.
   * @param \Drupal\ai_llms_txt_generator\Service\AiGeneratorService $ai_generator
   *   The AI generator service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    SitemapParserService $sitemap_parser,
    AiGeneratorService $ai_generator,
  ) {
    $this->configFactory = $config_factory;
    $this->sitemapParser = $sitemap_parser;
    $this->aiGenerator = $ai_generator;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('ai_llms_txt_generator.sitemap_parser'),
      $container->get('ai_llms_txt_generator.ai_generator')
    );
  }

  /**
   * Generate LLMs.txt content.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with generation status.
   */
  public function generate(Request $request) {
    $config = $this->configFactory->getEditable('ai_llms_txt_generator.settings');
    $sitemap_url = $config->get('sitemap_url');

    if (empty($sitemap_url)) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Sitemap URL not configured.',
      ], 400);
    }

    // Parse sitemap.
    $urls = $this->sitemapParser->parseSitemap($sitemap_url);

    if (empty($urls)) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Failed to parse sitemap or no URLs found.',
      ], 400);
    }

    // Format for AI.
    $sitemap_data = $this->sitemapParser->formatForAi($urls);

    // Generate content with AI.
    $generated_content = $this->aiGenerator->generateContent($sitemap_data);

    if ($generated_content === NULL) {
      return new JsonResponse([
        'success' => FALSE,
        'message' => 'Failed to generate content with AI.',
      ], 500);
    }

    // Save generated content.
    $config->set('generated_content', $generated_content)
      ->set('last_generated', time())
      ->save();

    return new JsonResponse([
      'success' => TRUE,
      'message' => 'LLMs.txt content generated successfully.',
      'content' => $generated_content,
    ]);
  }

}
