<?php

namespace Drupal\ai_migration\Plugin\migrate_plus\data_parser;

use Drupal\ai_migration\AiMigrator;
use Drupal\ai_migration\Service\HtmlContentProcessor;
use Drupal\ai_migration\Service\HtmlContentProcessorInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\migrate_plus\DataFetcherPluginManager;
use Drupal\migrate_plus\DataParserPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Obtain HTML for migration.
 *
 * @DataParser(
 *   id = "ai",
 *   title = @Translation("AI")
 * )
 */
class Ai extends DataParserPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The currently saved source url (as a string).
   *
   * @var string
   */
  protected $currentUrl;

  /**
   * The active url's source data.
   *
   * @var array
   */
  protected $sourceData;

  /**
   * The fetched items thus far.
   *
   * Note there should usually be one item per URL
   * so the count of this array should never exceed 1.
   *
   * @var array
   */
  protected $fetchedItems;

  /**
   * The migration object.
   *
   * @var \Drupal\migrate\MigrationInterface
   */
  protected $migration;

  /**
   * The results of parsing the fetched HTML with AI.
   *
   * @var array
   */
  protected $aiResults;

  /**
   * The AI migrator service.
   *
   * @var \Drupal\ai_migration\AiMigrator
   */
  protected $aiMigrator;

  /**
   * The HTML content processor service.
   *
   * @var \Drupal\ai_migration\Service\HtmlContentProcessorInterface
   */
  protected $htmlContentProcessor;

  /**
   * Creates a new Ai instance.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param string $plugin_definition
   *   The plugin definition.
   * @param \Drupal\migrate_plus\DataFetcherPluginManager $fetcherPluginManager
   *   The data fetcher plugin manager.
   * @param \Drupal\ai_migration\AiMigrator $aiMigrator
   *   The AI migrator service.
   * @param \Drupal\ai_migration\Service\HtmlContentProcessorInterface $htmlContentProcessor
   *   The HTML content processor service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    DataFetcherPluginManager $fetcherPluginManager,
    AiMigrator $aiMigrator,
    HtmlContentProcessorInterface $htmlContentProcessor,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $fetcherPluginManager);
    $this->urls = $configuration['urls'];
    $this->itemSelector = $configuration['item_selector'] ?? '';
    $this->aiMigrator = $aiMigrator;
    $this->htmlContentProcessor = $htmlContentProcessor;

    if (isset($configuration[$plugin_id][HtmlContentProcessor::MIGRATION_ROOT_KEY])) {
      $this->htmlContentProcessor->setConfig($configuration[$plugin_id][HtmlContentProcessor::MIGRATION_ROOT_KEY]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new Ai(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('plugin.manager.migrate_plus.data_fetcher'),
      $container->get('ai_migration.ai_migrator'),
      $container->get('ai_migration.html_content_processor')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function openSourceUrl(string $url): bool {
    // (Re)open the provided URL.
    // Use cached source data if this is the first request or URL is same as the
    // last time we made the request.
    if ($this->currentUrl != $url || !$this->sourceData) {
      // Use the data fetcher plugin to get the HTML content.
      $this->sourceData = $this->getDataFetcherPlugin()->getResponseContent($url);
      $this->currentUrl = $url;

      // Do we need to allow for multiple types?
      [$entity_type, $bundle] = explode(':', $this->configuration['types'][0]);

      if ($this->htmlContentProcessor->isConfigSet()) {
        $this->sourceData = $this->htmlContentProcessor->processHTML($this->sourceData);
      }

      // Now we can use the AI migrator to convert the HTML to structured data.
      $this->aiResults = $this->aiMigrator->convert($this->currentUrl, $this->sourceData, $entity_type, $bundle, $this->itemSelector);
    }

    // Ensure there is source data at the given URL and we have AI results.
    return !is_null($this->sourceData) && !empty($this->aiResults);
  }

  /**
   * {@inheritdoc}
   */
  protected function fetchNextRow(): void {
    if (empty($this->fetchedItems[$this->currentUrl])) {
      // The url is required since we're using it as a source key.
      $this->currentItem = [
        'url' => $this->currentUrl,
        ... $this->aiResults,
      ];

      // We've structured our AI query to return data
      // in a schema that works with Drupal.
      // We grab just the entity we need from the AI results.
      if (!empty($this->itemSelector)) {
        $this->currentItem = [
          'url' => $this->currentUrl,
          ... $this->aiResults[$this->itemSelector],
        ];
      }

      // We store the fetched item to ensure we don't fetch it again.
      $this->fetchedItems[$this->currentUrl] = $this->currentItem;
    }
  }

}
