<?php

declare(strict_types=1);

namespace Drupal\ai_migration\Service;

use Drupal\ai_migration\HtmlSanitizerConfigBuilderInterface;
use Symfony\Component\HtmlSanitizer\HtmlSanitizerInterface;

/**
 * Interface for HTML processing services.
 */
interface HtmlContentProcessorInterface {

  const MIGRATION_ROOT_KEY = 'html_processor';

  /**
   * Sets the HTML sanitizer instance.
   *
   * @param \Symfony\Component\HtmlSanitizer\HtmlSanitizerInterface $htmlSanitizer
   *   The HTML sanitizer to use.
   */
  public function setSanitizer(HtmlSanitizerInterface $htmlSanitizer): void;

  /**
   * Gets the current HTML sanitizer instance.
   *
   * @return \Symfony\Component\HtmlSanitizer\HtmlSanitizerInterface
   *   The HTML sanitizer or NULL if not set.
   */
  public function getSanitizer(): HtmlSanitizerInterface;

  /**
   * Sets the HTML sanitizer config builder.
   *
   * @param \Drupal\ai_migration\HtmlSanitizerConfigBuilderInterface $configBuilder
   *   The HTML sanitizer config builder to use.
   */
  public function setConfigBuilder(HtmlSanitizerConfigBuilderInterface $configBuilder): void;

  /**
   * Gets the current HTML sanitizer config builder.
   *
   * @return \Drupal\ai_migration\HtmlSanitizerConfigBuilderInterface
   *   The HTML sanitizer config builder.
   */
  public function getConfigBuilder(): HtmlSanitizerConfigBuilderInterface;

  /**
   * Check if the HTML processor config values are set.
   */
  public function isConfigSet(): bool;

  /**
   * Get the HTML processor config that was set.
   */
  public function getConfig(): mixed;

  /**
   * Set the HTML processor config values from the migration configuration.
   *
   * @param array $config
   *   The migration configuration.
   */
  public function setConfig(array $config = []): void;

  /**
   * Process HTML content based on migration configuration.
   *
   * @param string $html
   *   The HTML content to process.
   * @param array $config
   *   Options for processing the HTML content.
   *
   * @return string
   *   The processed HTML content.
   */
  public function processHtml(string $html, array $config = []): string;

}
