<?php

namespace Drupal\Tests\ai_migration\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\NodeType;
use Drupal\ai_migration\AiMigrator;

/**
 * Tests the AiMigrator class.
 *
 * @group ai_migration
 */
class AiMigratorTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'node',
    'file',
    'user',
    'serialization',
    'jsonapi',
    'ai',
    'ai_migration',
    'schemata',
    'schemata_json_schema',
  ];

  /**
   * The AI provider plugin manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $aiProviderPluginManager;

  /**
   * The AiMigrator instance being tested.
   *
   * @var \Drupal\ai_migration\AiMigrator
   */
  protected $aiMigrator;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install required configuration.
    $this->installConfig(['ai', 'ai_migration']);

    // Add article content type.
    $this->installEntitySchema('node');
    $node_type = NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ]);
    $node_type->save();

    // Get the AI provider plugin manager service.
    $this->aiProviderPluginManager = $this->container->get('ai.provider');

    // Get the logger service.
    $loggerChannelFactory = $this->container->get('logger.factory');

    // Get the HTTP client factory service.
    $httpClientFactory = $this->container->get('http_client_factory');

    // Get the cache service.
    $cache = $this->container->get('ai_migration.cache_provider');

    // Get the serializer service.
    $serializer = $this->container->get('serializer');

    // Get the schemata schema factory service.
    $schemaFactory = $this->container->get('schemata.schema_factory');

    // Create the AiMigrator instance.
    $this->aiMigrator = new AiMigrator($this->aiProviderPluginManager, $cache, $loggerChannelFactory, $httpClientFactory, $serializer, $schemaFactory);
  }

  /**
   * Tests the createPrompt method.
   */
  public function testCreatePrompt(): void {
    // Test with HTML content.
    $html = '<div>Test content</div>';

    $url = 'https://example.com/page';
    $result = $this->aiMigrator->createPrompt($url, $html);

    $expected = <<<PROMPT
    You can use HTML within the structure that you return,
    but do not place HTML <head> or <body> tags in any element of the schema.

    For any urls that are returned for a field in the schema, make sure they
    are absolute and not relative. This would apply to links or images.
    As an example of relative to absolute conversion, if the image href is
    /image1.jpg it would become https://example.com/image1.jpg
    if hostname is https://example.com and if image href is sub/image1.jpg
    when page url is https://example.com/level the url returned would be
    https://example.com/level/sub/image1.jpg.

    For any images where path is similar to https://example.com/sites/default/files/styles/max_width_288px/public/2022-09/IET.PNG?itok=3PK-CwYZ,
    refer to /styles/max_width_288px/public/ as "Drupal style path" of the URL (regex pattern would look like "/\/styles\/[^\/]+\/public\//").
    Remove the query string and the "Drupal style path" when returning the URL. For the example,
    the returned image URL would be https://example.com/sites/default/files/2022-09/IET.PNG.

    For any Unix timestamps that are returned, make sure they are in seconds and not milliseconds.

    For "created" and "changed" dates, use the current date and time in unix timestamp format.
    ';

    Here is the html:
    $html
PROMPT;

    $this->assertEquals($expected, $result, 'The createPrompt method should return the correct prompt');
  }

  /**
   * Tests the createSchema method.
   */
  public function testCreateSchema(): void {
    // Build a schema for our bundle.
    $entity_type = 'node';
    $bundle = 'article';

    $schema = $this->aiMigrator->createSchema($entity_type, $bundle);
    $json_schema = json_decode($schema, TRUE);

    $this->assertIsString($schema, 'Schema should be returned as a string');

    // Top-level schema metadata.
    $this->assertArrayHasKey('type', $json_schema, 'Schema should have a type field');
    $this->assertEquals('object', $json_schema['type'], 'Schema should be of type object');

    // Check on fields.
    $this->assertArrayHasKey('properties', $json_schema, 'Schema should have properties');
    $properties = $json_schema['properties']['data']['properties']['attributes']['properties'];

    // There should be a title field in the schema.
    $this->assertArrayHasKey('title', $properties, 'Schema should have title field');
    $this->assertArrayHasKey('type', $properties['title'], 'Title should have type');
    $this->assertEquals('string', $properties['title']['type'], 'Title should be string type');
  }

  /**
   * Tests the normalizeResponse method.
   */
  public function testNormalizeResponse(): void {
    $bundle = 'article';

    // Test with json code block markers.
    $response = "```json\n{\"data\": {\"attributes\": {\"title\": \"Test Title\"}}}\n```";
    $result = $this->aiMigrator->normalizeResponse($response, $bundle);
    $this->assertIsArray($result, 'Response should be converted to array');
    $this->assertEquals('Test Title', $result['title'], 'JSON should be properly decoded');

    // Test with backticks.
    $response = "`{\"data\": {\"attributes\": {\"title\": \"Another Title\"}}}`";
    $result = $this->aiMigrator->normalizeResponse($response, $bundle);
    $this->assertIsArray($result, 'Response should be converted to array');
    $this->assertEquals('Another Title', $result['title'], 'JSON with backticks should be properly decoded');

    // Test with clean JSON.
    $response = '{"data": {"attributes": {"title": "Clean Title"}}}';
    $result = $this->aiMigrator->normalizeResponse($response, $bundle);
    $this->assertIsArray($result, 'Response should be converted to array');
    $this->assertEquals('Clean Title', $result['title'], 'Clean JSON should be properly decoded');

    // Test with JSON with incorrect structure.
    $response = '{"title": "Clean Title"}';
    $result = $this->aiMigrator->normalizeResponse($response, $bundle);
    $this->assertIsBool($result, 'Response should be FALSE');
    $this->assertEquals(FALSE, $result, 'Improperly structured JSON should return FALSE');
  }

}
